"use client"

import { useState, useEffect } from "react"
import { ThumbsUp, ThumbsDown, Star, ExternalLink, Search, Package } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip"
import { useToast } from "@/components/ui/use-toast"
import Header from "@/components/Header"
import Footer from "@/components/Footer"
import Image from "next/image"
import { loadBrands, loadProducts, handleVote, getLocalStorageVote, loadCategories, loadFacilities } from '@/lib/firebase'
import type { Brand, Product, Category } from "@/types"
import MobileBottomBar from "@/components/MobileBottomBar"
import ProductGrid from "@/components/ProductGrid"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { ProductCard } from "@/components/ProductCard"

export default function MarkaPuanlamasiPage() {
  const [brands, setBrands] = useState<Brand[]>([])
  const [products, setProducts] = useState<Product[]>([])
  const [facilities, setFacilities] = useState<any[]>([])
  const [userVotes, setUserVotes] = useState<Record<string, "like" | "dislike" | null>>({})
  const [loading, setLoading] = useState(true)
  const [voting, setVoting] = useState<Record<string, boolean>>({})
  const [searchQuery, setSearchQuery] = useState("")
  const [searchResults, setSearchResults] = useState<{ brands: Brand[], products: Product[] }>({ brands: [], products: [] })
  const [showSearchResults, setShowSearchResults] = useState(false)
  const { toast } = useToast()
  const [categories, setCategories] = useState<Category[]>([])
  const [selectedBrand, setSelectedBrand] = useState<Brand | null>(null)

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true)
        const [brandsData, productsData, categoriesData, facilitiesData] = await Promise.all([
          loadBrands(),
          loadProducts(),
          loadCategories(),
          loadFacilities()
        ])
        
        const brandsWithAvgRating = brandsData.map(brand => {
          const brandProducts = productsData.filter(p => (typeof p.brand === 'string' ? p.brand : p.brand?.name) === brand.name);
          const totalRating = brandProducts.reduce((acc, p) => acc + (p.rating || 0), 0);
          const avgRating = brandProducts.length > 0 ? totalRating / brandProducts.length : 0;
          return { ...brand, stars: avgRating };
        });

        setBrands(brandsWithAvgRating as Brand[]);
        setProducts(productsData as Product[])
        setCategories(categoriesData as Category[])
        setFacilities(facilitiesData)
        
        // Her marka için localStorage'dan oylama durumunu al
        const votes: Record<string, "like" | "dislike" | null> = {}
        brandsData.forEach(brand => {
          votes[brand.id] = getLocalStorageVote('brand', brand.id)
        })
        setUserVotes(votes)
      } catch (error) {
        console.error("Veriler yüklenirken hata:", error)
      } finally {
        setLoading(false)
      }
    }

    fetchData()
  }, [])

  // Arama fonksiyonu
  const handleSearch = (query: string) => {
    setSearchQuery(query)
    
    if (!query.trim()) {
      setShowSearchResults(false)
      setSearchResults({ brands: [], products: [] })
      return
    }

    const lowerQuery = query.toLowerCase()
    
    // Markalarda arama
    const matchingBrands = brands.filter(brand => 
      brand.name.toLowerCase().includes(lowerQuery)
    )
    
    // Ürünlerde arama
    const matchingProducts = products.filter(product => 
      product.name.toLowerCase().includes(lowerQuery) ||
      (typeof product.brand === 'string' ? product.brand.toLowerCase().includes(lowerQuery) : product.brand?.name?.toLowerCase().includes(lowerQuery))
    )
    
    setSearchResults({ brands: matchingBrands, products: matchingProducts })
    setShowSearchResults(true)
  }

  const handleVoteClick = async (brandId: string, voteType: "like" | "dislike") => {
    if (voting[brandId]) return // Çift tıklamayı önle
    
    setVoting(prev => ({ ...prev, [brandId]: true }))
    
    try {
      const result = await handleVote('brand', brandId, voteType)
      
      if (result.success) {
        setUserVotes(prev => ({ ...prev, [brandId]: result.currentVote }))
        
        setBrands(prevBrands => prevBrands.map(brand => {
          if (brand.id === brandId) {
            return { ...brand, likes: result.likes, dislikes: result.dislikes };
          }
          return brand;
        }));

        toast({
          title: "Oylama",
          description: result.message,
          duration: 2000,
        })
      } else {
        toast({
          title: "Hata",
          description: result.message,
          variant: "destructive",
          duration: 3000,
        })
      }
    } catch (error) {
      console.error('Oylama hatası:', error)
      toast({
        title: "Hata",
        description: "Oylama kaydedilirken bir hata oluştu",
        variant: "destructive",
        duration: 3000,
      })
    } finally {
      setVoting(prev => ({ ...prev, [brandId]: false }))
    }
  }

  const getLikeRatio = (likes: number, dislikes: number) => {
    const total = likes + dislikes
    if (total === 0) return 0
    
    // Beğeni oranını 0-100 arasında hesapla
    const ratio = (likes / total) * 100
    
    // 0-100 arasında sınırla
    return Math.max(0, Math.min(100, ratio))
  }

  const getCategoryName = (categoryId: string) => {
    const category = categories.find(c => c.id === categoryId);
    return category ? category.name : categoryId;
  }

  const getCountryFlag = (countryCode: string) => {
    const flagEmojis: Record<string, string> = {
      TR: "🇹🇷",
      US: "🇺🇸",
      DE: "🇩🇪",
      FR: "🇫🇷",
      IT: "🇮🇹",
      ES: "🇪🇸",
      GB: "🇬🇧",
      CA: "🇨🇦",
      AU: "🇦🇺",
      JP: "🇯🇵",
      KR: "🇰🇷",
      CN: "🇨🇳",
      IN: "🇮🇳",
      BR: "🇧🇷",
      RU: "🇷🇺",
    }
    return flagEmojis[countryCode] || "🌍"
  }

  const displayBrands = showSearchResults ? searchResults.brands : brands

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      <Header onSearch={handleSearch} onCategoryChange={() => {}} showCategories={false} categories={categories} products={products} brands={brands} facilities={facilities} />

      <main className="flex-1 container mx-auto px-4 py-8">
        <div className="text-center mb-8">
          <h1 className="text-3xl font-bold text-gray-900 mb-4">Marka Puanlaması</h1>
          <p className="text-gray-600">En güvenilir vitamin ve takviye markalarını keşfedin</p>
        </div>

        {/* Arama Sonuçları */}
        {showSearchResults && (
          <div className="mb-8">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-xl font-semibold text-gray-900">
                "{searchQuery}" için arama sonuçları
              </h2>
              <Button variant="outline" onClick={() => setShowSearchResults(false)}>
                Tümünü Göster
              </Button>
            </div>
            
            {/* Marka Sonuçları */}
            {searchResults.brands.length > 0 && (
              <div className="mb-6">
                <h3 className="text-lg font-medium text-gray-700 mb-3 flex items-center gap-2">
                  <Star className="h-5 w-5 text-yellow-500" />
                  Markalar ({searchResults.brands.length})
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                  {searchResults.brands.map((brand) => {
                    const likeRatio = getLikeRatio(brand.likes || 0, brand.dislikes || 0)
                    const userVote = userVotes[brand.id]
                    const isTurkish = brand.countryCode === "TR"

                    return (
                      <Card key={brand.id} className="hover:shadow-lg transition-shadow">
                        <div className="aspect-video bg-gray-100 flex items-center justify-center relative">
                          <Image
                            src={brand.logoUrl || "/placeholder-logo.png"}
                            alt={brand.name}
                            width={200}
                            height={140}
                            className="object-contain p-4"
                            onError={(e) => {
                              const target = e.target as HTMLImageElement
                              target.src = "/placeholder-logo.png"
                            }}
                          />
                          
                          {/* Country Flag Badge */}
                          <div className="absolute top-2 right-2">
                            <Badge variant="secondary" className="text-sm">
                              {getCountryFlag(brand.countryCode || "TR")} {brand.countryCode || "TR"}
                            </Badge>
                          </div>

                          {/* Turkish Brand Badge */}
                          {isTurkish && (
                            <div className="absolute top-2 left-2">
                              <Badge variant="default" className="bg-red-600 text-white text-sm">
                                🇹🇷 Yerli & Milli
                              </Badge>
                            </div>
                          )}
                        </div>

                        <CardContent className="p-4">
                          <h3 className="text-lg font-semibold text-blue-600 mb-2">{brand.name}</h3>

                          {/* Star Rating */}
                          <div className="flex items-center gap-1 mb-3">
                            {[...Array(5)].map((_, i) => (
                              <Star
                                key={i}
                                className={`h-4 w-4 ${
                                  i < Math.floor(brand.stars || 0) 
                                    ? "fill-yellow-400 text-yellow-400" 
                                    : i === Math.floor(brand.stars || 0) && (brand.stars || 0) % 1 > 0
                                    ? "fill-yellow-400 text-yellow-400 opacity-60"
                                    : "text-gray-300"
                                }`}
                              />
                            ))}
                            <span className="text-sm text-gray-600 ml-1">({(brand.stars || 0).toFixed(1)})</span>
                          </div>

                          
                        </CardContent>
                      </Card>
                    )
                  })}
                </div>
              </div>
            )}

            {/* Ürün Sonuçları */}
            {searchResults.products.length > 0 && (
              <div>
                <h3 className="text-lg font-medium text-gray-700 mb-3 flex items-center gap-2">
                  <Package className="h-5 w-5 text-blue-500" />
                  Ürünler ({searchResults.products.length})
                </h3>
                <div className="space-y-3">
                  {searchResults.products.map((product) => (
                    <Card key={product.id} className="hover:shadow-md transition-shadow">
                      <CardContent className="p-4">
                        <div className="flex items-center gap-4">
                          <div className="w-16 h-16 bg-gray-100 rounded-lg flex items-center justify-center flex-shrink-0">
                            <Image
                              src={product.image || "/placeholder.jpg"}
                              alt={product.name}
                              width={64}
                              height={64}
                              className="object-contain rounded-lg"
                              onError={(e) => {
                                const target = e.target as HTMLImageElement
                                target.src = "/placeholder.jpg"
                              }}
                            />
                          </div>
                          <div className="flex-1 min-w-0">
                            <h4 className="font-medium text-gray-900 truncate">{product.name}</h4>
                            <p className="text-sm text-gray-600">
                              {typeof product.brand === 'string' ? product.brand : product.brand?.name}
                            </p>
                            <div className="flex items-center gap-2 mt-1">
                              <Badge variant="outline" className="text-xs">
                                {getCategoryName(product.category)}
                              </Badge>
                              {product.subCategory && (
                                <Badge variant="outline" className="text-xs">
                                  {product.subCategory}
                                </Badge>
                              )}
                            </div>
                          </div>
                          <div className="flex items-center gap-1 text-sm text-gray-600">
                            <Star className="h-4 w-4 fill-yellow-400 text-yellow-400" />
                            {product.rating?.toFixed(1) || "N/A"}
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </div>
            )}

            {/* Sonuç yoksa */}
            {searchResults.brands.length === 0 && searchResults.products.length === 0 && (
              <div className="text-center py-8">
                <p className="text-gray-500">"{searchQuery}" için sonuç bulunamadı.</p>
              </div>
            )}
          </div>
        )}

        {/* Normal Marka Listesi */}
        {!showSearchResults && (
          <>
        {loading ? (
          <div className="flex items-center justify-center py-20">
            <div className="text-center">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
              <p className="text-gray-600">Markalar yükleniyor...</p>
            </div>
          </div>
        ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                {displayBrands.map((brand) => {
              const likeRatio = getLikeRatio(brand.likes || 0, brand.dislikes || 0)
              const userVote = userVotes[brand.id]
              const isTurkish = brand.countryCode === "TR"

              return (
                <Card key={brand.id} className="hover:shadow-lg transition-shadow cursor-pointer" onClick={() => setSelectedBrand(brand)}>
                  <div className="aspect-video bg-gray-100 flex items-center justify-center relative">
                    <Image
                      src={brand.logoUrl || "/placeholder-logo.png"}
                      alt={brand.name}
                      width={200}
                      height={140}
                      className="object-contain p-4"
                      onError={(e) => {
                        const target = e.target as HTMLImageElement
                        target.src = "/placeholder-logo.png"
                      }}
                    />
                    
                    {/* Country Flag Badge */}
                    <div className="absolute top-2 right-2">
                      <Badge variant="secondary" className="text-sm">
                        {getCountryFlag(brand.countryCode || "TR")} {brand.countryCode || "TR"}
                      </Badge>
                    </div>

                    {/* Turkish Brand Badge */}
                    {isTurkish && (
                      <div className="absolute top-2 left-2">
                        <Badge variant="default" className="bg-red-600 text-white text-sm">
                          🇹🇷 Yerli & Milli
                        </Badge>
                      </div>
                    )}
                  </div>

                  <CardContent className="p-6">
                    <h3 className="text-xl font-semibold text-blue-600 mb-3">{brand.name}</h3>

                    {/* Star Rating */}
                    <div className="flex items-center gap-1 mb-4">
                      {[...Array(5)].map((_, i) => (
                        <Star
                          key={i}
                          className={`h-5 w-5 ${
                            i < Math.floor(brand.stars || 3) 
                              ? "fill-yellow-400 text-yellow-400" 
                              : i === Math.floor(brand.stars || 3) && (brand.stars || 3) % 1 > 0
                              ? "fill-yellow-400 text-yellow-400 opacity-60"
                              : "text-gray-300"
                          }`}
                        />
                      ))}
                      <span className="text-sm text-gray-600 ml-2">({(brand.stars || 3).toFixed(1)})</span>
                    </div>

                    {/* Bio List */}
                    <div className="text-sm text-gray-600 mb-4 space-y-2">
                      {brand.bio && brand.bio.map((bioItem, index) => (
                        <div key={index} className="flex items-start gap-2">
                          <span className="text-blue-600 mt-1">•</span>
                          <div className="flex-1">
                            <span>{bioItem.content}</span>
                            {bioItem.reference && (
                              <TooltipProvider>
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <a
                                      href={bioItem.reference}
                                      target="_blank"
                                      rel="noopener noreferrer"
                                      className="inline-flex items-center gap-1 ml-2 text-blue-600 hover:text-blue-800"
                                    >
                                      <ExternalLink className="h-3 w-3" />
                                    </a>
                                  </TooltipTrigger>
                                  <TooltipContent>
                                    <p>Kaynağı görüntüle</p>
                                  </TooltipContent>
                                </Tooltip>
                              </TooltipProvider>
                            )}
                          </div>
                        </div>
                      ))}
                    </div>

                    
                  </CardContent>
                </Card>
              )
            })}
          </div>
            )}
          </>
        )}
      </main>

      <Footer />
      <MobileBottomBar
        onSearch={handleSearch}
        onCategoryChange={() => {}}
        filters={{
          category: "*",
          subCategories: [] as string[],
          brands: [] as string[],
          contents: [] as string[],
          rating: { green: false, red: false },
          search: "",
          yerliMarka: false,
        }}
        onFiltersChange={() => {}}
        products={products}
        categories={categories}
        showFilterButton={false}
      />

      {selectedBrand && (
        <Dialog open={!!selectedBrand} onOpenChange={() => setSelectedBrand(null)}>
          <DialogContent className="max-w-4xl">
            <DialogHeader>
              <DialogTitle>{selectedBrand.name} Ürünleri</DialogTitle>
            </DialogHeader>
            <div className="max-h-[70vh] overflow-y-auto">
              <ProductGrid 
                products={products.filter(p => (typeof p.brand === 'string' ? p.brand : p.brand?.name) === selectedBrand.name)}
                categories={categories}
              />
            </div>
          </DialogContent>
        </Dialog>
      )}
    </div>
  )
}
