"use client";

import React from "react";
import { Product } from "@/types";
import { Card, CardContent } from "@/components/ui/card";
import Image from "next/image";
import { Button } from "@/components/ui/button";
import { Star, Award, Eye, Sparkles } from "lucide-react";
import { cn } from "@/lib/utils";
import Link from "next/link";
import Adsense from "@/components/Adsense";

interface TopQualityProductProps {
  product: Product;
  bgColor?: string;
  onProductClick?: (product: Product) => void;
  widgetTags?: string[];
  productType?: string;
  showAdsense?: boolean;
}

const TopQualityProduct: React.FC<TopQualityProductProps> = ({
  product,
  bgColor,
  widgetTags,
  productType = "Bitkisel Takviye",
  showAdsense,
}) => {
  return (
    <Link href={`/product/${product.id}`} className="block w-full h-full group">
      <div className={cn(
        "relative overflow-hidden rounded-xl shadow-lg bg-gradient-to-br from-blue-50 via-white to-purple-50 border border-gray-200 h-full min-h-[200px] sm:min-h-[250px]",
        bgColor
      )}>
        {/* Background Decorations */}
        <div className="absolute top-2 right-2 opacity-20">
          <Award size={32} className="text-yellow-500" />
        </div>
        <div className="absolute bottom-2 left-2 opacity-15">
          <Sparkles size={28} className="text-blue-500" />
        </div>

        {/* AdSense */}
        {showAdsense && (
          <div className="absolute left-1/2 -translate-x-1/2 bottom-4 z-20">
            <Adsense
              adClient="ca-pub-YOUR_ADSENSE_PUBLISHER_ID"
              adSlot="YOUR_ADSENSE_AD_UNIT_ID"
            />
          </div>
        )}

        {/* Content */}
        <div className="relative z-10 h-full flex flex-col p-4 sm:p-6">
          {/* Header */}
          <div className="text-center mb-4">
            <div className="flex items-center justify-center gap-2 mb-2">
              <Star className="h-5 w-5 text-yellow-500 fill-current" />
              <h2 className="text-base sm:text-lg font-bold text-gray-900">
                En Kaliteli Ürün
              </h2>
              <Star className="h-5 w-5 text-yellow-500 fill-current" />
            </div>
            <p className="text-xs sm:text-sm text-gray-600 max-w-xs mx-auto">
              Kullanıcılarımızın ve uzmanların favorisi
            </p>
          </div>

          {/* Product Card */}
          <div className="flex-1 flex items-center justify-center">
            <Card className="w-full max-w-sm bg-white/95 backdrop-blur-sm shadow-lg rounded-xl border-0 group-hover:scale-105 transition-all duration-300">
              <CardContent className="p-3 sm:p-4">
                {/* Product Image */}
                <div className="relative w-full h-24 sm:h-32 mb-3 bg-gray-50 rounded-lg overflow-hidden">
                  <Image
                    src={product.image || "/placeholder.svg"}
                    alt={product.name}
                    fill
                    className="object-contain p-2 group-hover:scale-110 transition-transform duration-300"
                    sizes="(max-width: 640px) 200px, 250px"
                  />

                  {/* Yerli Badge */}
                  {(product.yerliMarka || (typeof product.brand === 'object' && product.brand.countryCode === 'TR')) && (
                    <div className="absolute top-1 right-1">
                      <div className="bg-red-100 text-red-800 text-[10px] font-medium px-1.5 py-0.5 rounded-full">
                        🇹🇷
                      </div>
                    </div>
                  )}
                </div>

                {/* Product Info */}
                <div className="space-y-2">
                  <h3 className="font-semibold text-sm sm:text-base text-gray-900 line-clamp-2 leading-tight">
                    {product.name}
                  </h3>

                  <div className="flex items-center justify-between text-xs text-gray-500">
                    <span>{productType}</span>
                    <div className="flex items-center gap-1">
                      <Star className="h-3 w-3 text-yellow-500 fill-current" />
                      <span className="font-medium">Kaliteli</span>
                    </div>
                  </div>

                  {/* Action Button */}
                  <Button className="w-full bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-medium py-2 px-4 rounded-lg flex items-center justify-center gap-2 text-sm transition-all duration-200 group-hover:shadow-lg">
                    <Eye className="h-4 w-4" />
                    Detayları Gör
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </Link>
  );
};

export default TopQualityProduct;
