import { useEffect, useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs"
import { Trash2, Plus, Edit, Check, X } from "lucide-react"
import { loadCategories, deleteCategory, addSubCategory, deleteSubCategory, updateSubCategory, loadProducts, saveProduct, saveCategory } from "@/lib/firebase"
import type { Category } from "@/types"

export default function CategoriesTab() {
  const [categories, setCategories] = useState<Category[]>([])
  const [loading, setLoading] = useState(true)
  const [newCategory, setNewCategory] = useState("")
  const [activeTab, setActiveTab] = useState<string | null>(null)
  const [subCategoryInputs, setSubCategoryInputs] = useState<Record<string, string>>({})
  const [editingCategory, setEditingCategory] = useState<string | null>(null)
  const [editingName, setEditingName] = useState("")
  const [editingSubCategory, setEditingSubCategory] = useState<{categoryId: string, subId: string} | null>(null)
  const [editingSubName, setEditingSubName] = useState("")

  useEffect(() => {
    fetchCategories()
  }, [])

  const fetchCategories = async () => {
    setLoading(true)
    const cats = await loadCategories()
    setCategories(cats)
    setLoading(false)
    if (cats.length > 0 && !activeTab) setActiveTab(cats[0].id)
  }

  const handleAddCategory = async () => {
    if (!newCategory.trim()) return
    const newCatId = Date.now().toString(); // Simple unique ID
    await saveCategory({ id: newCatId, name: newCategory.trim(), subCategories: [] });
    setNewCategory("")
    fetchCategories()
  }

  const handleDeleteCategory = async (id: string) => {
    if (!window.confirm("Bu ana kategoriyi silmek istediğinize emin misiniz?")) return
    await deleteCategory(id)
    fetchCategories()
  }

  const handleEditCategory = (category: Category) => {
    setEditingCategory(category.id)
    setEditingName(category.name)
  }

  const handleSaveCategory = async (categoryId: string) => {
    if (!editingName.trim()) return
    const categoryToUpdate = categories.find(cat => cat.id === categoryId);
    if (categoryToUpdate) {
      await saveCategory({ ...categoryToUpdate, name: editingName.trim() });
    }
    setEditingCategory(null)
    setEditingName("")
    fetchCategories()
  }

  const handleCancelEdit = () => {
    setEditingCategory(null)
    setEditingName("")
  }

  const handleEditSubCategory = (categoryId: string, subCategory: {id: string, name: string}) => {
    setEditingSubCategory({categoryId, subId: subCategory.id})
    setEditingSubName(subCategory.name)
  }

  const handleSaveSubCategory = async () => {
    if (!editingSubCategory || !editingSubName.trim()) return
    await updateSubCategory(editingSubCategory.categoryId, { id: editingSubCategory.subId, name: editingSubName.trim() })
    setEditingSubCategory(null)
    setEditingSubName("")
    fetchCategories()
  }

  const handleCancelSubEdit = () => {
    setEditingSubCategory(null)
    setEditingSubName("")
  }

  const handleAddSubCategory = async (categoryId: string) => {
    const name = subCategoryInputs[categoryId]?.trim()
    if (!name) return
    const newSubCatId = Date.now().toString(); // Simple unique ID
    await addSubCategory(categoryId, { id: newSubCatId, name: name });
    setSubCategoryInputs((prev) => ({ ...prev, [categoryId]: "" }))
    fetchCategories()
  }

  const handleDeleteSubCategory = async (categoryId: string, subCategoryId: string) => {
    if (!window.confirm("Bu alt kategoriyi silmek istediğinize emin misiniz?")) return
    await deleteSubCategory(categoryId, subCategoryId)
    fetchCategories()
  }

  const handleSyncCategories = async () => {
    const [products, categories] = await Promise.all([loadProducts(), loadCategories()]);
    let changed = false;
    const categoryMap = new Map(categories.map(cat => [cat.name.toLowerCase(), cat]));
    for (const product of products) {
      const catName = product.category?.trim();
      const subCatName = product.subCategory?.trim();
      if (!catName) continue;
      let category = categoryMap.get(catName.toLowerCase());
      if (!category) {
        const newCatId = Date.now().toString();
        await saveCategory({ id: newCatId, name: catName, subCategories: [] });
        changed = true;
        const updatedCategories = await loadCategories();
        category = updatedCategories.find(cat => cat.id === newCatId);
        if (category) categoryMap.set(catName.toLowerCase(), category);
      }
      if (category && subCatName && !category.subCategories.some(sc => sc.name.toLowerCase() === subCatName.toLowerCase())) {
        await addSubCategory(category.id, subCatName);
        changed = true;
      }
    }
    if (changed) fetchCategories();
  };

  const handleDeleteAllCategories = async () => {
    if (!window.confirm("Tüm kategorileri silmek istediğinize emin misiniz? Bu işlem geri alınamaz ve ürünlerin kategori referansları boşa düşer!")) return;
    const cats = await loadCategories();
    for (const cat of cats) {
      await deleteCategory(cat.id);
    }
    fetchCategories();
  };

  const handleFixProductCategories = async () => {
    const [products, categories] = await Promise.all([loadProducts(), loadCategories()]);
    let updated = 0;
    for (const product of products) {
      // Kategori adıysa düzelt
      if (product.category && !categories.some(cat => cat.id === product.category)) {
        const match = categories.find(cat => cat.name.trim().toLowerCase() === product.category.trim().toLowerCase());
        if (match) {
          await saveProduct({ ...product, category: match.id });
          updated++;
        }
      }
      // yerliMarka alanı düzeltmesi (string ise boolean yap)
      if (typeof product.yerliMarka !== 'boolean') {
        const yerli = product.yerliMarka === 'true' || product.yerliMarka === true;
        await saveProduct({ ...product, yerliMarka: yerli });
        updated++;
      }
    }
    alert(`${updated} ürün güncellendi!`);
  };

  return (
    <div>
      <Card className="mb-6">
        <CardHeader>
          <CardTitle>Kategori Yönetimi</CardTitle>
          <Button onClick={handleSyncCategories} variant="secondary" className="ml-4">Yenile (Ürünlerden Kategori Ekle)</Button>
          <Button onClick={handleDeleteAllCategories} variant="destructive" className="ml-4">Tüm Kategorileri Sil</Button>
          <Button onClick={handleFixProductCategories} variant="outline" className="ml-4">Ürün Kategorilerini Düzelt</Button>
        </CardHeader>
        <CardContent>
          <div className="flex gap-2 items-center">
            <Input
              placeholder="Yeni ana kategori adı"
              value={newCategory}
              onChange={(e) => setNewCategory(e.target.value)}
              className="max-w-xs"
            />
            <Button onClick={handleAddCategory} variant="default">
              <Plus className="h-4 w-4 mr-1" /> Ana Kategori Ekle
            </Button>
          </div>
        </CardContent>
      </Card>

      {loading ? (
        <div className="text-center py-8 text-gray-500">Yükleniyor...</div>
      ) : categories.length === 0 ? (
        <div className="text-center py-8 text-gray-500">Hiç ana kategori yok.</div>
      ) : (
        <Tabs value={activeTab || undefined} onValueChange={setActiveTab} className="w-full">
          <TabsList className="mb-4">
            {categories.map((cat) => (
              <TabsTrigger key={cat.id} value={cat.id}>
                {cat.name}
              </TabsTrigger>
            ))}
          </TabsList>
          {categories.map((cat) => (
            <TabsContent key={cat.id} value={cat.id}>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between">
                  <div className="flex items-center gap-2">
                    {editingCategory === cat.id ? (
                      <div className="flex items-center gap-2">
                        <Input
                          value={editingName}
                          onChange={(e) => setEditingName(e.target.value)}
                          className="w-48"
                          autoFocus
                        />
                        <Button size="sm" onClick={() => handleSaveCategory(cat.id)} variant="default">
                          <Check className="h-4 w-4" />
                        </Button>
                        <Button size="sm" onClick={handleCancelEdit} variant="outline">
                          <X className="h-4 w-4" />
                        </Button>
                      </div>
                    ) : (
                      <CardTitle className="flex items-center gap-2">
                        {cat.name}
                        <Button size="sm" variant="ghost" onClick={() => handleEditCategory(cat)}>
                          <Edit className="h-4 w-4" />
                        </Button>
                      </CardTitle>
                    )}
                  </div>
                  <Button variant="destructive" size="icon" onClick={() => handleDeleteCategory(cat.id)}>
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </CardHeader>
                <CardContent>
                  <div className="mb-4">
                    <div className="flex gap-2 items-center">
                      <Input
                        placeholder="Yeni alt kategori adı"
                        value={subCategoryInputs[cat.id] || ""}
                        onChange={(e) => setSubCategoryInputs((prev) => ({ ...prev, [cat.id]: e.target.value }))}
                        className="max-w-xs"
                      />
                      <Button onClick={() => handleAddSubCategory(cat.id)} variant="secondary">
                        <Plus className="h-4 w-4 mr-1" /> Alt Kategori Ekle
                      </Button>
                    </div>
                  </div>
                  <div>
                    <h4 className="font-semibold mb-2">Alt Kategoriler</h4>
                    {cat.subCategories.length === 0 ? (
                      <div className="text-gray-400">Hiç alt kategori yok.</div>
                    ) : (
                      <ul className="space-y-2">
                        {cat.subCategories.map((sub) => (
                          <li key={sub.id} className="flex items-center justify-between bg-gray-100 rounded px-3 py-2">
                            {editingSubCategory?.categoryId === cat.id && editingSubCategory?.subId === sub.id ? (
                              <div className="flex items-center gap-2 flex-1">
                                <Input
                                  value={editingSubName}
                                  onChange={(e) => setEditingSubName(e.target.value)}
                                  className="flex-1"
                                  autoFocus
                                />
                                <Button size="sm" onClick={handleSaveSubCategory} variant="default">
                                  <Check className="h-3 w-3" />
                                </Button>
                                <Button size="sm" onClick={handleCancelSubEdit} variant="outline">
                                  <X className="h-3 w-3" />
                                </Button>
                              </div>
                            ) : (
                              <>
                                <span>{sub.name}</span>
                                <div className="flex items-center gap-1">
                                  <Button 
                                    variant="ghost" 
                                    size="icon" 
                                    onClick={() => handleEditSubCategory(cat.id, sub)}
                                    className="h-6 w-6"
                                  >
                                    <Edit className="h-3 w-3" />
                                  </Button>
                                  <Button 
                                    variant="ghost" 
                                    size="icon" 
                                    onClick={() => handleDeleteSubCategory(cat.id, sub.id)}
                                    className="h-6 w-6 text-red-500"
                                  >
                                    <Trash2 className="h-3 w-3" />
                                  </Button>
                                </div>
                              </>
                            )}
                          </li>
                        ))}
                      </ul>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          ))}
        </Tabs>
      )}
    </div>
  )
} 