"use client";

import React, { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { CustomWidgetTemplate, Product } from "@/types";
import { loadCustomWidgetTemplates, saveCustomWidgetTemplate, deleteCustomWidgetTemplate, loadProducts } from "@/lib/firebase"; // These will be updated to generic widget functions
import ProductSelect from "./ProductSelect";
import WidgetTagSelect from "./WidgetTagSelect";
import { toast } from "@/components/ui/use-toast";

export function CustomWidgetsTab() {
  const [widgetTemplates, setWidgetTemplates] = useState<CustomWidgetTemplate[]>([]);
  const [products, setProducts] = useState<Product[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [currentWidgetTemplate, setCurrentWidgetTemplate] = useState<CustomWidgetTemplate | null>(null);
  const [templateName, setTemplateName] = useState("");
  const [templateType, setTemplateType] = useState<CustomWidgetTemplate["type"]>("product_carousel");
  const [selectedProductIds, setSelectedProductIds] = useState<string[]>([]);
  const [selectedProductId, setSelectedProductId] = useState<string | undefined>(undefined);
  const [widgetTitle, setWidgetTitle] = useState("");
  const [widgetDescription, setWidgetDescription] = useState("");
  const [widgetBgColor, setWidgetBgColor] = useState("");
  const [selectedWidgetTags, setSelectedWidgetTags] = useState<string[]>([]);
  const [widgetW, setWidgetW] = useState<number>(1); // New state for width
  const [widgetH, setWidgetH] = useState<number>(1); // New state for height
  const [infoContent, setInfoContent] = useState<string>(""); // For info_card
  const [sourceLink, setSourceLink] = useState<string>(""); // For info_card
  const [infoImageUrl, setInfoImageUrl] = useState<string>(""); // Yeni: info_card görsel url
  const [showAdsense, setShowAdsense] = useState<boolean>(false); // Adsense kutucuğu

  const availableWidgetTags = ["Doktor Onayı", "Kaliteli", "Güvenli", "En İyi 100 Ürün", "İlk 10"];

  useEffect(() => {
    fetchWidgetTemplates();
    fetchProducts();
  }, []);

  const fetchWidgetTemplates = async () => {
    // Assuming loadCustomCarousels will be updated to loadCustomWidgetTemplates
    const data = await loadCustomWidgetTemplates() as CustomWidgetTemplate[]; 
    setWidgetTemplates(data);
    console.log("Fetched Widget Templates:", data); // Debugging log
  };

  const fetchProducts = async () => {
    const data = await loadProducts();
    setProducts(data);
  };

  const handleAddWidgetTemplate = () => {
    setCurrentWidgetTemplate(null);
    setTemplateName("");
    setTemplateType("product_carousel");
    setSelectedProductIds([]);
    setSelectedProductId(undefined);
    setWidgetTitle("");
    setWidgetDescription("");
    setWidgetBgColor("");
    setSelectedWidgetTags([]);
    setWidgetW(5); // Default for product_carousel
    setWidgetH(2); // Default for product_carousel
    setInfoContent("");
    setSourceLink("");
    setInfoImageUrl("");
    setShowAdsense(false);
    setIsModalOpen(true);
  };

  const handleEditWidgetTemplate = (template: CustomWidgetTemplate) => {
    setCurrentWidgetTemplate(template);
    setTemplateName(template.name);
    setTemplateType(template.type);
    setSelectedProductIds(template.productIds || []);
    setSelectedProductId(template.productId);
    setWidgetTitle(template.title || "");
    setWidgetDescription(template.description || "");
    setWidgetBgColor(template.bgColor || "");
    setSelectedWidgetTags(template.widgetTags || []);
    setWidgetW(template.w || 1); // Load existing width, default to 1
    setWidgetH(template.h || 1); // Load existing height, default to 1
    setInfoContent(template.infoContent || "");
    setSourceLink(template.sourceLink || "");
    setInfoImageUrl(template.infoImageUrl || "");
    setShowAdsense(template.showAdsense ?? false);
    setIsModalOpen(true);
  };

  const handleDeleteWidgetTemplate = async (id: string) => {
    if (window.confirm("Bu widget şablonunu silmek istediğinizden emin misiniz?")) {
      try {
        // Assuming deleteCustomCarousel will be updated to deleteCustomWidgetTemplate
        await deleteCustomWidgetTemplate(id);
        fetchWidgetTemplates();
        toast({
          title: "Başarılı",
          description: "Widget şablonu başarıyla silindi.",
        });
      } catch (error) {
        console.error("Widget şablonu silinirken hata oluştu:", error);
        toast({
          title: "Hata",
          description: "Widget şablonu silinirken bir hata oluştu.",
          variant: "destructive",
        });
      }
    }
  };

  const handleSaveWidgetTemplate = async () => {
    if (!templateName.trim()) {
      toast({
        title: "Hata",
        description: "Şablon adı boş olamaz.",
        variant: "destructive",
      });
      return;
    }

    const widgetToSave: CustomWidgetTemplate = {
      id: currentWidgetTemplate?.id || Date.now().toString(),
      name: templateName.trim(),
      type: templateType,
      title: widgetTitle.trim() || undefined,
      description: widgetDescription.trim() || undefined,
      bgColor: widgetBgColor.trim() || "",
      widgetTags: selectedWidgetTags,
      w: widgetW,
      h: widgetH,
      showAdsense: showAdsense,
    };

    if (templateType === "product_carousel") {
      widgetToSave.productIds = selectedProductIds;
    } else if (templateType === "single_product") {
      widgetToSave.productId = selectedProductId;
    } else if (templateType === "info_card") {
      widgetToSave.infoContent = infoContent.trim();
      widgetToSave.sourceLink = sourceLink.trim();
      widgetToSave.infoImageUrl = infoImageUrl.trim();
    }

    try {
      // Assuming saveCustomCarousel will be updated to saveCustomWidgetTemplate
      await saveCustomWidgetTemplate(widgetToSave as any);
      fetchWidgetTemplates();
      setIsModalOpen(false);
      toast({
        title: "Başarılı",
        description: "Widget şablonu başarıyla kaydedildi.",
      });
    } catch (error) {
      console.error("Widget şablonu kaydedilirken hata oluştu:", error);
      toast({
        title: "Hata",
        description: "Widget şablonu kaydedilirken bir hata oluştu.",
        variant: "destructive",
      });
    }
  };

  return (
    <div className="p-4">
      <div className="flex justify-between items-center mb-4">
        <h2 className="text-2xl font-semibold">Özelleştirilmiş Widget Şablonları</h2>
        <Button onClick={handleAddWidgetTemplate}>Yeni Widget Şablonu Ekle</Button>
      </div>

      <Table>
        <TableHeader>
          <TableRow>
            <TableHead>Şablon Adı</TableHead>
            <TableHead>Tipi</TableHead>
            <TableHead>Başlık</TableHead>
            <TableHead className="text-right">İşlemler</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {widgetTemplates.map((template) => (
            <TableRow key={template.id}>
              <TableCell className="font-medium">{template.name}</TableCell>
              <TableCell>{template.type === "product_carousel" ? "Ürün Karoseli" : "Tek Ürün"}</TableCell>
              <TableCell>{template.title || "-"}</TableCell>
              <TableCell className="text-right flex items-center justify-end space-x-2">
                <Button
                  variant="outline"
                  size="sm"
                  className="mr-2"
                  onClick={() => handleEditWidgetTemplate(template)}
                >
                  Düzenle
                </Button>
                <Button
                  variant="destructive"
                  size="sm"
                  onClick={() => handleDeleteWidgetTemplate(template.id)}
                >
                  Sil
                </Button>
              </TableCell>
            </TableRow>
          ))}
        </TableBody>
      </Table>

      <Dialog open={isModalOpen} onOpenChange={setIsModalOpen}>
        <DialogContent className="sm:max-w-[600px]">
          <DialogHeader>
            <DialogTitle>{currentWidgetTemplate ? "Widget Şablonunu Düzenle" : "Yeni Widget Şablonu Ekle"}</DialogTitle>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="showAdsense" className="text-right">
                Adsense Gösterilsin mi?
              </Label>
              <input
                id="showAdsense"
                type="checkbox"
                checked={showAdsense}
                onChange={e => setShowAdsense(e.target.checked)}
                className="col-span-3 w-5 h-5"
              />
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="templateName" className="text-right">
                Şablon Adı
              </Label>
              <Input
                id="templateName"
                value={templateName}
                onChange={(e) => setTemplateName(e.target.value)}
                className="col-span-3"
              />
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="templateType" className="text-right">
                Şablon Tipi
              </Label>
              <Select onValueChange={(value: CustomWidgetTemplate["type"]) => {
                setTemplateType(value);
                if (value === "product_carousel") {
                  setWidgetW(4);
                  setWidgetH(1);
                } else if (value === "single_product") {
                  setWidgetW(2);
                  setWidgetH(1);
                } else if (value === "info_card") {
                  setWidgetW(2);
                  setWidgetH(2);
                }
              }} value={templateType}>
                <SelectTrigger className="col-span-3">
                  <SelectValue placeholder="Şablon Tipi Seç" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="product_carousel">Ürün Karoseli</SelectItem>
                  <SelectItem value="single_product">Tek Ürün</SelectItem>
                  <SelectItem value="info_card">Bilgi Kartı</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="widgetTitle" className="text-right">
                Widget Başlığı
              </Label>
              <Input
                id="widgetTitle"
                value={widgetTitle}
                onChange={(e) => setWidgetTitle(e.target.value)}
                className="col-span-3"
                placeholder="Örn: Doktorların En Sevdiği Ürünler"
              />
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="widgetDescription" className="text-right">
                Açıklama
              </Label>
              <Input
                id="widgetDescription"
                value={widgetDescription}
                onChange={(e) => setWidgetDescription(e.target.value)}
                className="col-span-3"
                placeholder="Örn: Uzmanlarımızın seçimi..."
              />
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="widgetBgColor" className="text-right">
                Arka Plan Teması
              </Label>
              <Select onValueChange={(value) => setWidgetBgColor(value)} value={widgetBgColor} >
                <SelectTrigger className="col-span-3">
                  <SelectValue placeholder="Arka Plan Teması Seç" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="default">Varsayılan (Beyaz)</SelectItem>
                  <SelectItem value="bg-gradient-to-r from-blue-50 to-green-50">Doktor Onaylı (Açık Mavi/Yeşil)</SelectItem>
                  <SelectItem value="bg-gradient-to-br from-white via-gray-50 to-[#f2f2f2]">En Kaliteli (Gri Tonları)</SelectItem>
                  <SelectItem value="bg-gradient-to-r from-orange-100 to-orange-200">Canlı Turuncu</SelectItem>
                  <SelectItem value="bg-gradient-to-r from-blue-100 to-blue-200">Sakin Mavi</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="widgetTags" className="text-right">
                Widget Etiketleri
              </Label>
              <div className="col-span-3">
                <WidgetTagSelect
                  availableTags={availableWidgetTags}
                  selectedTags={selectedWidgetTags}
                  onTagsChange={setSelectedWidgetTags}
                />
              </div>
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="widgetW" className="text-right">
                Genişlik (w)
              </Label>
              <Input
                id="widgetW"
                type="number"
                value={widgetW}
                onChange={(e) => setWidgetW(parseInt(e.target.value))}
                className="col-span-3"
                min={1}
              />
            </div>

            <div className="grid grid-cols-4 items-center gap-4">
              <Label htmlFor="widgetH" className="text-right">
                Yükseklik (h)
              </Label>
              <Input
                id="widgetH"
                type="number"
                value={widgetH}
                onChange={(e) => setWidgetH(parseInt(e.target.value))}
                className="col-span-3"
                min={1}
              />
            </div>

            {templateType === "info_card" && (
              <>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="infoContent" className="text-right">
                    Bilgi İçeriği
                  </Label>
                  <Input
                    id="infoContent"
                    value={infoContent}
                    onChange={(e) => setInfoContent(e.target.value)}
                    className="col-span-3"
                    placeholder="Bilgi kartının ana içeriği"
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="infoImageUrl" className="text-right">
                    Görsel URL
                  </Label>
                  <Input
                    id="infoImageUrl"
                    value={infoImageUrl}
                    onChange={(e) => setInfoImageUrl(e.target.value)}
                    className="col-span-3"
                    placeholder="https://..."
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="sourceLink" className="text-right">
                    Kaynak Linki
                  </Label>
                  <Input
                    id="sourceLink"
                    value={sourceLink}
                    onChange={(e) => setSourceLink(e.target.value)}
                    className="col-span-3"
                    placeholder="Kaynak linki (isteğe bağlı)"
                  />
                </div>
              </>
            )}

            {templateType === "product_carousel" && (
              <div className="grid grid-cols-4 items-start gap-4">
                <Label htmlFor="products" className="text-right pt-2">
                  Ürünler
                </Label>
                <div className="col-span-3">
                  <ProductSelect
                    allProducts={products}
                    selectedProductIds={selectedProductIds}
                    onProductIdsChange={setSelectedProductIds}
                  />
                </div>
              </div>
            )}

            {templateType === "single_product" && (
              <div className="grid grid-cols-4 items-start gap-4">
                <Label htmlFor="singleProduct" className="text-right pt-2">
                  Ürün Seç
                </Label>
                <div className="col-span-3">
                  <Select onValueChange={(value) => setSelectedProductId(value)} value={selectedProductId}>
                    <SelectTrigger>
                      <SelectValue placeholder="Tek Ürün Seç" />
                    </SelectTrigger>
                    <SelectContent>
                      {products.map(p => <SelectItem key={p.id} value={p.id}>{p.name}</SelectItem>)}
                    </SelectContent>
                  </Select>
                </div>
              </div>
            )}
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsModalOpen(false)}>
              İptal
            </Button>
            <Button onClick={handleSaveWidgetTemplate}>Kaydet</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}