"use client";

import React, { useCallback, useEffect, useState } from "react";
import { Product } from "@/types";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import Image from "next/image";
import { cn } from "@/lib/utils";
import useEmblaCarousel from "embla-carousel-react";
import Autoplay from "embla-carousel-autoplay";
import { ArrowLeft, ArrowRight, Stethoscope } from "lucide-react"; // Stethoscope for icon, assuming heart rate is CSS
import { Badge } from "@/components/ui/badge";
import { ProductCard } from "@/components/ProductCard";

interface DoctorApprovedProductsCarouselProps {
  products: Product[];
  bgColor?: string;
  onProductClick: (product: Product) => void;
  widgetTags?: string[];
  categories?: Array<{ id: string; name: string }>
}

const DoctorApprovedProductsCarousel: React.FC<DoctorApprovedProductsCarouselProps> = ({ products, bgColor, onProductClick, widgetTags, categories = [] }) => {
  const [emblaRef, emblaApi] = useEmblaCarousel({ loop: true, containScroll: 'trimSnaps' }, [Autoplay({ delay: 3000, stopOnInteraction: false })]);
  const [prevBtnDisabled, setPrevBtnDisabled] = useState(true);
  const [nextBtnDisabled, setNextBtnDisabled] = useState(true);
  const [selectedIndex, setSelectedIndex] = useState(0);
  const [scrollSnaps, setScrollSnaps] = useState<number[]>([]);

  const scrollPrev = useCallback(() => emblaApi && emblaApi.scrollPrev(), [emblaApi]);
  const scrollNext = useCallback(() => emblaApi && emblaApi.scrollNext(), [emblaApi]);
  const scrollTo = useCallback((index: number) => emblaApi && emblaApi.scrollTo(index), [emblaApi]);

  const onInit = useCallback((emblaApi: any) => {
    setScrollSnaps(emblaApi.scrollSnapList());
  }, []);

  const onSelect = useCallback((emblaApi: any) => {
    setSelectedIndex(emblaApi.selectedScrollSnap());
    setPrevBtnDisabled(!emblaApi.canScrollPrev());
    setNextBtnDisabled(!emblaApi.canScrollNext());
  }, []);

  useEffect(() => {
    if (!emblaApi) return;

    onInit(emblaApi);
    onSelect(emblaApi);
    emblaApi.on("reInit", onInit);
    emblaApi.on("reInit", onSelect);
    emblaApi.on("select", onSelect);
  }, [emblaApi, onInit, onSelect]);

  return (
    <section className={cn("relative py-12 overflow-hidden", bgColor || "bg-gradient-to-r from-blue-50 to-green-50")}>
      {/* Background Visual Elements */}
      <div className="absolute bottom-0 right-0 opacity-20">
        <Stethoscope size={200} className="text-blue-300" />
      </div>
      <div className="absolute top-0 left-0 opacity-10 blur-sm">
        {/* Placeholder for heartbeat graph - can be an SVG or more complex CSS */}
        <div className="w-48 h-24 bg-green-300 rounded-full transform -rotate-45"></div>
      </div>

      <div className="container mx-auto px-4 relative z-10 max-w-screen-lg">
        <h2 className="text-center text-lg font-semibold text-gray-800 mb-8 text-left flex items-center gap-2">
          <Stethoscope className="h-6 w-6 text-blue-600" />
          Doktorların En Sevdiği Ürünler
        </h2>

        <div className="embla">
          <div className="embla__viewport" ref={emblaRef}>
            <div className="embla__container flex">
              {products.map((product) => (
                <div className="embla__slide flex-[0_0_50%] sm:flex-[0_0_33.333%] md:flex-[0_0_25%] px-2" key={product.id}>
                  <ProductCard product={product} onClick={() => onProductClick(product)} categories={categories} />
                </div>
              ))}
            </div>
          </div>

          <div className="embla__dots flex justify-center mt-4">
            {scrollSnaps.map((_, index) => (
              <button
                key={index}
                className={cn(
                  "embla__dot w-3 h-3 rounded-full mx-1 bg-gray-300 transition-colors duration-200",
                  index === selectedIndex && "bg-blue-500"
                )}
                onClick={() => scrollTo(index)}
              />
            ))}
          </div>

          <div className="embla__buttons absolute top-1/2 -translate-y-1/2 left-0 right-0 flex justify-between px-4">
            <Button
              className="embla__button embla__button--prev bg-white text-gray-800 rounded-full p-2 shadow-md hover:bg-gray-100"
              onClick={scrollPrev}
              disabled={prevBtnDisabled}
            >
              <ArrowLeft size={24} />
            </Button>
            <Button
              className="embla__button embla__button--next bg-white text-gray-800 rounded-full p-2 shadow-md hover:bg-gray-100"
              onClick={scrollNext}
              disabled={nextBtnDisabled}
            >
              <ArrowRight size={24} />
            </Button>
          </div>
        </div>
      </div>
    </section>
  );
};

export default DoctorApprovedProductsCarousel;
