"use client"

import type { Product, Category } from "@/types"
import { ProductCard } from "./ProductCard"
import { Button } from "@/components/ui/button"
import { Loader2 } from "lucide-react"

interface ProductGridProps {
  products: Product[]
  loading?: boolean
  onProductClick?: (product: Product) => void
  onRefresh?: () => Promise<void>
  categories?: Category[]
  brands?: any[]
  onLoadMore?: () => void
  hasMore?: boolean
}

export default function ProductGrid({
  products,
  loading,
  onProductClick,
  onRefresh,
  categories = [],
  brands = [],
  onLoadMore,
  hasMore
}: ProductGridProps) {

  if (loading) {
    return (
      <div className="flex items-center justify-center py-16">
        <div className="text-center">
          <Loader2 className="h-8 w-8 animate-spin text-blue-600 mx-auto mb-4" />
          <p className="text-gray-600 text-sm">Ürünler yükleniyor...</p>
        </div>
      </div>
    )
  }

  if (products.length === 0) {
    return (
      <div className="text-center py-16 px-4">
        <div className="max-w-md mx-auto">
          <div className="mb-4">
            <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <span className="text-2xl">🔍</span>
            </div>
          </div>
          <h3 className="text-lg font-semibold text-gray-900 mb-2">
            Ürün Bulunamadı
          </h3>
          <p className="text-gray-600 text-sm">
            Seçili filtrelere uygun ürün bulunamadı. Lütfen filtrelerinizi değiştirerek tekrar deneyin.
          </p>
        </div>
      </div>
    )
  }

  return (
    <div className="space-y-6">
      {/* Mobile: 2x2 Grid, Desktop: 4 columns */}
      <div className="grid grid-cols-2 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-5 gap-3 sm:gap-4">
        {products.map((product) => (
          <div key={product.id} className="h-full">
            <ProductCard
              product={product}
              onRefresh={onRefresh}
              categories={categories}
              brands={brands}
            />
          </div>
        ))}
      </div>

      {/* Load More Button */}
      {hasMore && (
        <div className="flex justify-center pt-6 pb-8">
          <Button
            onClick={onLoadMore}
            className="px-8 py-3 text-base font-medium bg-blue-600 hover:bg-blue-700 text-white rounded-full shadow-lg hover:shadow-xl transition-all duration-200 transform hover:scale-105"
            size="lg"
          >
            Daha Fazla Ürün Yükle
          </Button>
        </div>
      )}
    </div>
  )
}
