"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { Upload, Star, Plus, Trash2 } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import type { Brand } from "@/types"
import type { Product, Facility } from "@/types"
import Image from "next/image"
import { saveBrand } from "@/lib/firebase"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { getAISafetyScore, getCleanContentScore, getCertificateScore, getTotalSafetyScore } from "@/lib/score-calculator"

interface BrandModalProps {
  brand: Brand | null
  onClose: () => void
  onSave: (brand: Brand) => void
  products: Product[]
  facilities: Facility[]
}

export default function BrandModal({ brand, onClose, onSave, products = [], facilities = [] }: BrandModalProps) {
  const [formData, setFormData] = useState({
    name: "",
    logoUrl: "",
    countryCode: "TR",
    stars: 3,
    likes: 0,
    dislikes: 0,
    bio: [] as Array<{ content: string; reference?: string }>,
    influencerCommission: [] as { platform: string; rate: number }[],
  })

  useEffect(() => {
    if (brand) {
      setFormData({
        name: brand.name || "",
        logoUrl: brand.logoUrl || "",
        countryCode: brand.countryCode || "TR",
        stars: brand.stars || 3,
        likes: brand.likes || 0,
        dislikes: brand.dislikes || 0,
        bio: brand.bio || [],
        influencerCommission: brand.influencerCommission || [],
      })
    }
  }, [brand])

  const [saving, setSaving] = useState(false)

  const addBioItem = () => {
    setFormData({
      ...formData,
      bio: [...formData.bio, { content: "", reference: "" }]
    })
  }

  const removeBioItem = (index: number) => {
    setFormData({
      ...formData,
      bio: formData.bio.filter((_, i) => i !== index)
    })
  }

  const updateBioItem = (index: number, field: 'content' | 'reference', value: string) => {
    const newBio = [...formData.bio]
    newBio[index] = { ...newBio[index], [field]: value }
    setFormData({
      ...formData,
      bio: newBio
    })
  }

  const addCommission = () => {
    setFormData({
      ...formData,
      influencerCommission: [...formData.influencerCommission, { platform: '', rate: 0 }]
    })
  }

  const removeCommission = (index: number) => {
    setFormData({
      ...formData,
      influencerCommission: formData.influencerCommission.filter((_, i) => i !== index)
    })
  }

  const updateCommission = (index: number, field: 'platform' | 'rate', value: string | number) => {
    const newCom = [...formData.influencerCommission]
    newCom[index] = { ...newCom[index], [field]: value }
    setFormData({ ...formData, influencerCommission: newCom })
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()

    try {
      setSaving(true)

      const brandData = {
        id: brand?.id,
        name: formData.name,
        logoUrl: formData.logoUrl,
        countryCode: formData.countryCode,
        stars: formData.stars,
        likes: formData.likes,
        dislikes: formData.dislikes,
        bio: formData.bio.filter(item => item.content.trim() !== ""),
        influencerCommission: formData.influencerCommission.filter(ic => ic.platform && ic.rate > 0),
      }

      const savedId = await saveBrand(brandData)
      console.log("Marka kaydedildi:", savedId)

      onSave({ ...brandData, id: savedId })
    } catch (error) {
      console.error("Marka kaydedilirken hata:", error)
      alert("Marka kaydedilirken bir hata oluştu.")
    } finally {
      setSaving(false)
    }
  }

  // Markanın ürettiği ürünler
  const brandProducts = products.filter(p => {
    if (!brand) return false;
    if (typeof p.brand === 'string') return p.brand.trim().toLowerCase() === brand.name.trim().toLowerCase();
    if (typeof p.brand === 'object' && p.brand !== null) return p.brand.id === brand.id || p.brand.name.trim().toLowerCase() === brand.name.trim().toLowerCase();
    return false;
  });

  // Kayıt no normalize fonksiyonu
  const normalizeKayitNo = (kod: string) => (kod || '').replace(/[^a-zA-Z0-9]/g, '').toLowerCase();

  // Markanın ürünlerinden çıkan tüm kayıt no'lar
  const productKayitNos = Array.from(new Set(brandProducts.map(p => normalizeKayitNo((p as any)["uretimTesisiKayitNo"] || (p as any)["Üretici İşletme Kayıt/Onay Numarası"] || '')))).filter(Boolean);

  // Bu kayıt no'lara sahip tesisler
  const relatedFacilities = facilities.filter(fac => productKayitNos.includes(normalizeKayitNo((fac as any).kayit_no || '')));

  return (
    <Dialog open={true} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[95vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>{brand ? "Marka Düzenle" : "Yeni Marka Ekle"}</DialogTitle>
        </DialogHeader>

        <form onSubmit={handleSubmit} className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Marka Bilgileri</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="name">Marka Adı *</Label>
                  <Input
                    id="name"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="countryCode">Ülke Kodu</Label>
                  <Select
                    value={formData.countryCode}
                    onValueChange={(value) => setFormData({ ...formData, countryCode: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Ülke seçin" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="TR">🇹🇷 Türkiye</SelectItem>
                      <SelectItem value="US">🇺🇸 Amerika Birleşik Devletleri</SelectItem>
                      <SelectItem value="DE">🇩🇪 Almanya</SelectItem>
                      <SelectItem value="FR">🇫🇷 Fransa</SelectItem>
                      <SelectItem value="JP">🇯🇵 Japonya</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="stars">Yıldız (1-5)</Label>
                <div className="flex items-center gap-2">
                  <Input
                    id="stars"
                    type="number"
                    min="1"
                    max="5"
                    step="0.1"
                    value={formData.stars}
                    onChange={(e) => setFormData({ ...formData, stars: Number.parseFloat(e.target.value) || 1 })}
                    className="w-20"
                  />
                  <div className="flex items-center gap-1">
                    {[...Array(5)].map((_, i) => (
                      <Star
                        key={i}
                        className={`h-4 w-4 ${
                          i < Math.floor(formData.stars) 
                            ? "fill-yellow-400 text-yellow-400" 
                            : i === Math.floor(formData.stars) && formData.stars % 1 > 0
                            ? "fill-yellow-400 text-yellow-400 opacity-60"
                            : "text-gray-300"
                        }`}
                      />
                    ))}
                  </div>
                </div>
                <p className="text-xs text-gray-500">Kesirli değerler kabul edilir (örn: 4.5)</p>
              </div>

              <div className="space-y-2">
                <Label htmlFor="logoUrl">Logo URL</Label>
                <div className="flex gap-2">
                  <Input
                    id="logoUrl"
                    type="url"
                    value={formData.logoUrl}
                    onChange={(e) => setFormData({ ...formData, logoUrl: e.target.value })}
                    placeholder="https://example.com/logo.jpg"
                    className="flex-1"
                  />
                  <Button 
                    type="button" 
                    variant="outline"
                    onClick={() => {
                      const url = prompt("Logo URL'sini girin:")
                      if (url) {
                        setFormData({ ...formData, logoUrl: url })
                      }
                    }}
                  >
                    <Upload className="h-4 w-4" />
                  </Button>
                </div>
                
                {/* Logo Önizleme */}
                <div className="mt-3">
                  <Label className="text-sm text-gray-600 mb-2 block">Logo Önizleme:</Label>
                  <div className="flex items-center gap-4">
                    <div className="w-20 h-20 bg-gray-100 border rounded-lg flex items-center justify-center overflow-hidden">
                      {formData.logoUrl ? (
                        <Image
                          src={formData.logoUrl}
                          alt="Logo önizleme"
                          width={80}
                          height={80}
                          className="object-contain w-full h-full"
                          onError={(e) => {
                            const target = e.target as HTMLImageElement
                            target.src = "/placeholder.svg"
                          }}
                        />
                      ) : (
                        <div className="text-gray-400 text-xs text-center">
                          Logo yok
                        </div>
                      )}
                    </div>
                    
                    {/* Hızlı Logo Seçenekleri */}
                    <div className="flex gap-2">
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => setFormData({ ...formData, logoUrl: "/placeholder-logo.png" })}
                      >
                        Placeholder
                      </Button>
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => setFormData({ ...formData, logoUrl: "" })}
                      >
                        Temizle
                      </Button>
                    </div>
                  </div>
                  
                  {/* URL Doğrulama */}
                  {formData.logoUrl && (
                    <div className="mt-2 text-xs">
                      {formData.logoUrl.startsWith('http') ? (
                        <span className="text-green-600">✓ Geçerli URL</span>
                      ) : (
                        <span className="text-red-600">⚠ Geçerli bir URL girin</span>
                      )}
                    </div>
                  )}
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="likes">Beğeni Sayısı</Label>
                  <Input
                    id="likes"
                    type="number"
                    min="0"
                    value={formData.likes}
                    onChange={(e) => setFormData({ ...formData, likes: Number.parseInt(e.target.value) || 0 })}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="dislikes">Beğenmeme Sayısı</Label>
                  <Input
                    id="dislikes"
                    type="number"
                    min="0"
                    value={formData.dislikes}
                    onChange={(e) => setFormData({ ...formData, dislikes: Number.parseInt(e.target.value) || 0 })}
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label>Influencer Komisyon Oranı</Label>
                <div className="space-y-2">
                  {formData.influencerCommission.map((item, idx) => (
                    <div key={idx} className="flex gap-2 items-center">
                      <Select
                        value={item.platform}
                        onValueChange={val => updateCommission(idx, 'platform', val)}
                      >
                        <SelectTrigger className="w-36">
                          <SelectValue placeholder="Platform seçin" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="Trendyol">Trendyol</SelectItem>
                          <SelectItem value="Hepsiburada">Hepsiburada</SelectItem>
                          <SelectItem value="Yerel Site">Yerel Site</SelectItem>
                        </SelectContent>
                      </Select>
                      <Input
                        type="number"
                        min="0"
                        max="100"
                        step="0.1"
                        className="w-24"
                        value={item.rate}
                        onChange={e => updateCommission(idx, 'rate', Number(e.target.value))}
                        placeholder="%"
                      />
                      <span className="text-gray-500">%</span>
                      <Button type="button" size="sm" variant="ghost" onClick={() => removeCommission(idx)}>
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}
                  <Button type="button" size="sm" variant="outline" onClick={addCommission}>
                    <Plus className="h-4 w-4 mr-1" /> Ekle
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Bio Section */}
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg">Marka Bilgileri & Referanslar</CardTitle>
                <Button type="button" onClick={addBioItem} variant="outline" size="sm">
                  <Plus className="h-4 w-4 mr-1" />
                  Ekle
                </Button>
              </div>
            </CardHeader>
            <CardContent className="space-y-4">
              {formData.bio.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  <p>Henüz marka bilgisi eklenmemiş</p>
                  <Button type="button" onClick={addBioItem} variant="outline" className="mt-2">
                    <Plus className="h-4 w-4 mr-1" />
                    İlk Bilgiyi Ekle
                  </Button>
                </div>
              ) : (
                formData.bio.map((item, index) => (
                  <div key={index} className="border rounded-lg p-4 space-y-3">
                    <div className="flex items-center justify-between">
                      <Label className="text-sm font-medium">Bilgi #{index + 1}</Label>
                      <Button
                        type="button"
                        onClick={() => removeBioItem(index)}
                        variant="outline"
                        size="sm"
                        className="text-red-600 hover:text-red-700"
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                    
                    <div className="space-y-2">
                      <Label htmlFor={`bio-content-${index}`}>İçerik *</Label>
                      <Textarea
                        id={`bio-content-${index}`}
                        value={item.content}
                        onChange={(e) => updateBioItem(index, 'content', e.target.value)}
                        placeholder="Marka hakkında bilgi veya yorum..."
                        rows={3}
                        required
                      />
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor={`bio-reference-${index}`}>Referans Linki (Opsiyonel)</Label>
                      <Input
                        id={`bio-reference-${index}`}
                        type="url"
                        value={item.reference || ""}
                        onChange={(e) => updateBioItem(index, 'reference', e.target.value)}
                        placeholder="https://example.com/reference"
                      />
                      <p className="text-xs text-gray-500">
                        Kaynak linki varsa ekleyin. Boş bırakılabilir.
                      </p>
                    </div>
                  </div>
                ))
              )}
            </CardContent>
          </Card>

          {/* Markanın ürettiği ürünler */}
          {brand && (
            <div className="mt-8">
              <h3 className="text-lg font-semibold mb-2">Ürettiği Ürünler</h3>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                {brandProducts.length === 0 && <div className="text-gray-400">Ürün bulunamadı.</div>}
                {brandProducts.map((p) => (
                  <div key={p.id} className="flex items-center gap-2 bg-gray-50 border rounded p-2 min-h-[64px]">
                    <div className="w-14 h-14 flex-shrink-0 bg-white rounded flex items-center justify-center overflow-hidden">
                      <Image src={p.image || "/placeholder.svg"} alt={p.name} width={56} height={56} className="object-contain" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="font-medium text-sm truncate" title={p.name}>{p.name}</div>
                      <div className="flex flex-col gap-0.5 mt-1">
                        <span className="text-xs text-blue-700">AI Güven: <b>{getAISafetyScore(p)}%</b></span>
                        <span className="text-xs text-green-700">Temiz İçerik: <b>{getCleanContentScore(p)}%</b></span>
                        <span className="text-xs text-yellow-700">Sertifika: <b>{getCertificateScore(p)}%</b></span>
                        <span className="text-xs text-purple-700">Popülerlik: <b>{p.likes + p.dislikes > 0 ? Math.round((p.likes / (p.likes + p.dislikes)) * 100) : 'N/A'}%</b></span>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}
          {/* İlgili üretim tesisleri */}
          {relatedFacilities.length > 0 && (
            <div className="mt-8">
              <h3 className="text-lg font-semibold mb-2">İlgili Üretim Tesisleri</h3>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                {relatedFacilities.map(fac => (
                  <div key={fac.id} className="flex items-center gap-2 bg-gray-50 border rounded p-2 min-h-[64px]">
                    <div className="w-12 h-12 flex-shrink-0 bg-white rounded flex items-center justify-center overflow-hidden">
                      <Image src={fac.logoUrl || "/placeholder-logo.png"} alt={fac.name} width={48} height={48} className="object-contain" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="font-medium text-sm truncate" title={fac.name}>{fac.name}</div>
                      <div className="text-xs text-gray-600">Kayıt No: {(fac as any).kayit_no || '-'}</div>
                      <div className="text-xs text-yellow-700">Puan: <b>{fac.score}</b></div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}
          {/* Form Actions */}
          <div className="flex justify-end gap-4">
            <Button type="button" variant="outline" onClick={onClose}>
              İptal
            </Button>
            <Button type="submit" disabled={saving}>
              {saving ? "Kaydediliyor..." : brand ? "Güncelle" : "Kaydet"}
            </Button>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  )
}
