"use client"

import type React from "react"
import { useState, useEffect } from "react"
import { Plus, Trash2 } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import type { Facility } from "@/types"
import type { Brand } from "@/types"
import { saveFacility } from "@/lib/firebase"
import { Select, SelectTrigger, SelectValue, SelectContent, SelectItem } from "@/components/ui/select"
import ProductGrid from "@/components/ProductGrid"
import { loadProducts } from "@/lib/firebase"

interface FacilityModalProps {
  facility: Facility | null
  onClose: () => void
  onSave: (facility: Facility) => void
  brands: Brand[]
  products?: any[]
}

const CERTS = [
  { key: "gmp", label: "GMP" },
  { key: "haccp", label: "HACCP" },
  { key: "iso9001", label: "ISO 9001" },
  { key: "iso22000", label: "ISO 22000" },
]

export default function FacilityModal({ facility, onClose, onSave, brands, products = [] }: FacilityModalProps) {
  const [formData, setFormData] = useState({
    name: "",
    logoUrl: "",
    score: 0,
    bio: [] as Array<{ content: string; reference?: string }>,
    certificates: { gmp: false, haccp: false, iso9001: false, iso22000: false },
    brandsProduced: [] as string[],
  })

  const [selectedBrand, setSelectedBrand] = useState<string>("")

  useEffect(() => {
    if (facility) {
      setFormData({
        name: facility.name || "",
        logoUrl: facility.logoUrl || "",
        score: facility.score || 0,
        bio: facility.bio || [],
        certificates: {
          gmp: facility.certificates?.gmp || false,
          haccp: facility.certificates?.haccp || false,
          iso9001: facility.certificates?.iso9001 || false,
          iso22000: facility.certificates?.iso22000 || false,
        },
        brandsProduced: facility.brandsProduced || [],
      })
    }
  }, [facility])

  const [saving, setSaving] = useState(false)

  const addBioItem = () => {
    setFormData({
      ...formData,
      bio: [...formData.bio, { content: "", reference: "" }]
    })
  }

  const removeBioItem = (index: number) => {
    setFormData({
      ...formData,
      bio: formData.bio.filter((_, i) => i !== index)
    })
  }

  const updateBioItem = (index: number, field: 'content' | 'reference', value: string) => {
    const newBio = [...formData.bio]
    newBio[index] = { ...newBio[index], [field]: value }
    setFormData({
      ...formData,
      bio: newBio
    })
  }

  const handleCertChange = (key: string, checked: boolean) => {
    setFormData({
      ...formData,
      certificates: { ...formData.certificates, [key]: checked }
    })
  }

  const handleAddBrandProduced = () => {
    if (selectedBrand && !formData.brandsProduced.includes(selectedBrand)) {
      setFormData((prev) => ({
        ...prev,
        brandsProduced: [...prev.brandsProduced, selectedBrand],
      }))
      setSelectedBrand("")
    }
  }

  const handleRemoveBrandProduced = (brandName: string) => {
    setFormData((prev) => ({
      ...prev,
      brandsProduced: prev.brandsProduced.filter((b) => b !== brandName),
    }))
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    try {
      setSaving(true)
      const facilityData = {
        id: facility?.id,
        name: formData.name,
        logoUrl: formData.logoUrl,
        score: Math.max(0, Math.min(100, Number(formData.score) || 0)),
        bio: formData.bio.filter(item => item.content.trim() !== ""),
        certificates: formData.certificates,
        brandsProduced: formData.brandsProduced,
      }
      const savedId = await saveFacility(facilityData)
      onSave({ ...facilityData, id: savedId })
    } catch (error) {
      console.error("Tesis kaydedilirken hata:", error)
      alert("Tesis kaydedilirken bir hata oluştu.")
    } finally {
      setSaving(false)
    }
  }

  // Ürettiği ürünleri bul
  function normalizeKayitNo(kod: string) {
    if (!kod || typeof kod !== 'string') return '';
    return kod.replace(/[^a-zA-Z0-9]/g, '').toLowerCase();
  }
  const facilityKayitNo = normalizeKayitNo(facility?.kayit_no || facility?.name || '');
  const producedProducts = products.filter(p => normalizeKayitNo(p["Üretici İşletme Kayıt/Onay Numarası"] || p.uretimTesisiKayitNo || '') === facilityKayitNo);

  return (
    <Dialog open={true} onOpenChange={onClose}>
      <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>{facility ? "Tesis Düzenle" : "Yeni Tesis Ekle"}</DialogTitle>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Tesis Bilgileri</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="name">Tesis Adı *</Label>
                  <Input
                    id="name"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="logoUrl">Logo URL</Label>
                  <Input
                    id="logoUrl"
                    value={formData.logoUrl}
                    onChange={(e) => setFormData({ ...formData, logoUrl: e.target.value })}
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="score">Tesis Puanı (0-100)</Label>
                <Input
                  id="score"
                  type="number"
                  min="0"
                  max="100"
                  value={formData.score}
                  onChange={(e) => setFormData({ ...formData, score: Math.max(0, Math.min(100, Number(e.target.value) || 0)) })}
                  className="w-32"
                  required
                />
                <p className="text-xs text-gray-500">100 üzerinden puan verin</p>
              </div>
              {/* Sertifikalar */}
              <div className="space-y-2">
                <Label>Sertifikalar</Label>
                <div className="flex flex-wrap gap-4">
                  {CERTS.map(cert => (
                    <label key={cert.key} className="flex items-center gap-2 text-sm">
                      <input
                        type="checkbox"
                        checked={!!formData.certificates[cert.key as keyof typeof formData.certificates]}
                        onChange={e => handleCertChange(cert.key, e.target.checked)}
                      />
                      {cert.label}
                    </label>
                  ))}
                </div>
              </div>
              {/* Üretilen Markalar */}
              <div className="space-y-2">
                <Label>Üretilen Markalar</Label>
                <div className="flex gap-2 items-center">
                  <Select value={selectedBrand} onValueChange={setSelectedBrand}>
                    <SelectTrigger className="w-64">
                      <SelectValue placeholder="Marka seçin" />
                    </SelectTrigger>
                    <SelectContent>
                      {brands.filter(b => !formData.brandsProduced.includes(b.name)).map((brand) => (
                        <SelectItem key={brand.id} value={brand.name}>{brand.name}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Button type="button" variant="outline" size="sm" onClick={handleAddBrandProduced} disabled={!selectedBrand}>
                    Ekle
                  </Button>
                </div>
                <div className="flex flex-wrap gap-2 mt-2">
                  {formData.brandsProduced.map((brandName) => (
                    <span key={brandName} className="inline-flex items-center bg-blue-100 text-blue-800 px-2 py-1 rounded text-xs">
                      {brandName}
                      <Button type="button" size="icon" variant="ghost" className="ml-1" onClick={() => handleRemoveBrandProduced(brandName)}>
                        <Trash2 className="h-3 w-3" />
                      </Button>
                    </span>
                  ))}
                </div>
                <p className="text-xs text-gray-500">Birden fazla marka ekleyebilirsiniz</p>
              </div>
              {/* Bio */}
              <div className="space-y-2">
                <Label>Biyografi / Açıklama</Label>
                <div className="space-y-2">
                  {formData.bio.map((item, idx) => (
                    <div key={idx} className="flex gap-2 items-start">
                      <Textarea
                        value={item.content}
                        onChange={e => updateBioItem(idx, 'content', e.target.value)}
                        placeholder="Açıklama"
                        className="flex-1"
                        rows={2}
                        required
                      />
                      <Input
                        value={item.reference || ""}
                        onChange={e => updateBioItem(idx, 'reference', e.target.value)}
                        placeholder="Referans (opsiyonel)"
                        className="w-48"
                      />
                      <Button type="button" variant="ghost" onClick={() => removeBioItem(idx)}>
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}
                  <Button type="button" variant="outline" size="sm" onClick={addBioItem}>
                    <Plus className="h-4 w-4 mr-1" /> Açıklama Ekle
                  </Button>
                </div>
              </div>
              {/* Ürettiği Ürünler */}
              {producedProducts.length > 0 && (
                <div className="mt-6">
                  <h3 className="text-lg font-semibold mb-2">Bu Tesiste Üretilen Ürünler</h3>
                  <ProductGrid products={producedProducts} loading={false} onProductClick={() => {}} />
                </div>
              )}
            </CardContent>
          </Card>
          <div className="flex justify-end gap-2">
            <Button type="button" variant="outline" onClick={onClose}>İptal</Button>
            <Button type="submit" disabled={saving}>{saving ? "Kaydediliyor..." : "Kaydet"}</Button>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  )
} 