"use client"

import { useState, useEffect } from "react"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Select, SelectContent, SelectTrigger, SelectValue, SelectItem } from "@/components/ui/select"
import { Checkbox } from "@/components/ui/checkbox"
import { Badge } from "@/components/ui/badge"
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command"
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover"
import { Sparkles, ChevronsUpDown, Check, Info } from "lucide-react"
import { saveProduct, loadBrands } from "@/lib/firebase"
import { getAISafetyScore, getCleanContentScore, getCertificateScore, getTotalSafetyScore } from "@/lib/score-calculator"
import type { Product, Brand, Category, Facility } from "@/types"
import { cn } from "@/lib/utils"
import { useIsMobile } from "@/hooks/use-mobile"
import Image from "next/image"

interface ProductModalProps {
  product: Product | null
  onClose: () => void
  onSave: (product: Product) => void
  categories: Category[]
  products: Product[]
  allBrands: Brand[]
  facilities: Facility[]
}

const subCategories: Record<string, string[]> = {
  "Sporcu Takviyeleri": ["Protein Tozları", "Amino Asitler", "Kreatin", "BCAA", "Glutamin"],
  "Çocuk Takviyeleri": ["Omega 3", "Vitamin Şurubu", "Bağışıklık Destekleyici", "Multivitamin"],
  "Vitamin & Mineral": ["C Vitamini", "D Vitamini", "Magnezyum", "B Kompleksi", "E Vitamini"],
  "Meyve Cipsleri": ["Elma Cipsi", "Muz Cipsi", "Çilek Cipsi", "Karışık Meyve Cipsi"],
  "Bitkisel Takviyeler": ["Yeşil Çay", "Zencefil", "Zerdeçal", "Ginseng", "Adaçayı"],
  "Bağışıklık Destekleyiciler": ["Propolis", "Beta Glukan", "Kolajen", "Çinko Takviyesi"],
}

export default function ProductModal({ product, onClose, onSave, categories = [], products = [], allBrands = [], facilities = [] }: ProductModalProps) {
  const [brands, setBrands] = useState<Brand[]>([])
  const [brandSearchOpen, setBrandSearchOpen] = useState(false)
  const [brandSearchValue, setBrandSearchValue] = useState("")
  const [formData, setFormData] = useState({
    name: "",
    brand: { id: "", name: "" },
    category: "",
    subCategory: "",
    price: "",
    image: "",
    buyLink: "",
    likes: 0,
    dislikes: 0,
    kaliteliHammadde: false,
    yerliMarka: false,
    ingredients: "",
    usage: "",
    dradvice: false,
    evaluation: {
      temiz_icerik: "",
      formul_kalitesi: "",
      uretim_tesisi: "",
      kullanici_yorumlari: "",
      uretim_sertifika: "",
      ai_guven_skoru: "",
    },
    cleanFactors: {
      form: {
        tablet: false,
        kapsul: false,
        toz: false,
        sase: false,
        gummy: false,
        sivi: false,
      },
      additives: {
        renklendirici: false,
        aroma: false,
        tatlandirici: false,
      },
    },
    certFactors: {
      gmp: false,
      haccp: false,
      iso9001: false,
      iso22000: false,
    },
    influencerCommission: [{ platform: "Trendyol", rate: 0 }],
  })

  const [aiScore, setAiScore] = useState(0)
  const [cleanScore, setCleanScore] = useState(0)
  const [certScore, setCertScore] = useState(0)
  const [saving, setSaving] = useState(false)

  // Markaları yükle
  useEffect(() => {
    const loadBrandsData = async () => {
      try {
        const brandsData = await loadBrands()
        setBrands(brandsData as Brand[])
      } catch (error) {
        console.error("Markalar yüklenirken hata:", error)
      }
    }
    loadBrandsData()
  }, [])

  useEffect(() => {
    if (product) {
      // Seçili markanın influencerCommission'unu bul
      let defaultCommissions: { platform: string; rate: number }[] = [];
      let initialBrand: { id: string; name: string };

      if (product.brand) {
        if (typeof product.brand === 'object' && product.brand !== null) {
          initialBrand = { id: product.brand.id || "", name: product.brand.name || "" };
        } else { // typeof product.brand === 'string'
          initialBrand = { id: "", name: product.brand };
        }
        const brandObj = allBrands?.find(b => b.id === initialBrand.id || b.name.trim().toLowerCase() === initialBrand.name.trim().toLowerCase());
        if (brandObj && Array.isArray(brandObj.influencerCommission)) {
          defaultCommissions = brandObj.influencerCommission;
        }
      } else {
        initialBrand = { id: "", name: "" };
      }

      setFormData({
        name: product.name || "",
        brand: initialBrand,
        category: product.category || "",
        subCategory: product.subCategory || "",
        price: product.price?.toString() || "",
        image: product.image || "",
        buyLink: product.buyLink || "",
        likes: product.likes || 0,
        dislikes: product.dislikes || 0,
        kaliteliHammadde: product.kaliteliHammadde || false,
        yerliMarka: product.yerliMarka || false,
        ingredients: product.icerik?.join(", ") || "",
        usage: product.usage || "",
        dradvice: product.evaluation?.dradvice || false,
        evaluation: {
          temiz_icerik: product.evaluation?.temiz_icerik?.toString() || "",
          formul_kalitesi: product.evaluation?.formul_kalitesi?.toString() || "",
          uretim_tesisi: product.evaluation?.uretim_tesisi?.toString() || "",
          kullanici_yorumlari: product.evaluation?.kullanici_yorumlari?.toString() || "",
          uretim_sertifika: product.evaluation?.uretim_sertifika?.toString() || "",
          ai_guven_skoru: product.evaluation?.ai_guven_skoru?.toString() || "",
        },
        cleanFactors: product.cleanFactors ? { ...product.cleanFactors } : {
          form: {
            tablet: false,
            kapsul: false,
            toz: false,
            sase: false,
            gummy: false,
            sivi: false,
          },
          additives: {
            renklendirici: false,
            aroma: false,
            tatlandirici: false,
          },
        },
        certFactors: product.evaluation?.certFactors ? { ...product.evaluation.certFactors } : {
          gmp: false,
          haccp: false,
          iso9001: false,
          iso22000: false,
        },
        influencerCommission: product.influencerCommission && product.influencerCommission.length > 0 ? product.influencerCommission : defaultCommissions,
      });
    } else {
      // Yeni ürün eklerken form verilerini sıfırla
      setFormData({
        name: "",
        brand: { id: "", name: "" },
        category: "",
        subCategory: "",
        price: "",
        image: "",
        buyLink: "",
        likes: 0,
        dislikes: 0,
        kaliteliHammadde: false,
        yerliMarka: false,
        ingredients: "",
        usage: "",
        dradvice: false,
        evaluation: {
          temiz_icerik: "",
          formul_kalitesi: "",
          uretim_tesisi: "",
          kullanici_yorumlari: "",
          uretim_sertifika: "",
          ai_guven_skoru: "",
        },
        cleanFactors: {
          form: {
            tablet: false,
            kapsul: false,
            toz: false,
            sase: false,
            gummy: false,
            sivi: false,
          },
          additives: {
            renklendirici: false,
            aroma: false,
            tatlandirici: false,
          },
        },
        certFactors: {
          gmp: false,
          haccp: false,
          iso9001: false,
          iso22000: false,
        },
        influencerCommission: [{ platform: "Trendyol", rate: 0 }],
      });
    }
  }, [product?.id]);

  // Calculate scores
  useEffect(() => {
    // AI Safety Score - Manuel girilen değer
    const aiPct = Number.parseInt(formData.evaluation.ai_guven_skoru) || 0
    if (aiScore !== aiPct) {
      setAiScore(aiPct)
    }

    // Clean Content Score - Form tipi ve katkı maddelerine göre
    let cleanPct = 0
    
    // Form tipi puanı (sadece bir tanesi seçili olmalı)
    const selectedForm = Object.entries(formData.cleanFactors.form).find(([_, isSelected]) => isSelected)
    if (selectedForm) {
      const [formType] = selectedForm
      const formScores: Record<string, number> = {
        sivi: 75,
        kapsul: 100,
        tablet: 90,
        toz: 80,
        sase: 85,
        gummy: 70,
      }
      cleanPct = formScores[formType] || 0
    }

    // Katkı maddesi cezaları
    Object.entries(formData.cleanFactors.additives).forEach(([additive, hasAdditive]) => {
      if (hasAdditive) {
        const penalties: Record<string, number> = {
          renklendirici: 20,
          aroma: 15,
          tatlandirici: 10,
        }
        const penalty = penalties[additive] || 0
        cleanPct = Math.max(0, cleanPct - penalty) // Negatif olmasın
      }
    })

    setCleanScore(cleanPct)

    // Certificate Score - Her sertifika %25 puan ekler
    const certCount = Object.values(formData.certFactors).filter(Boolean).length
    const certPct = certCount * 25
    setCertScore(certPct)
  }, [JSON.stringify(formData.cleanFactors), JSON.stringify(formData.certFactors), formData.evaluation.ai_guven_skoru])

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()

    try {
      setSaving(true)

      // Rating hesapla
      const calculatedRating = formData.likes + formData.dislikes > 0 
        ? Math.round((formData.likes / (formData.likes + formData.dislikes)) * 5 * 10) / 10
        : 0

      const productData = {
        id: product?.id,
        name: formData.name,
        brand: formData.brand,
        category: formData.category,
        subCategory: formData.subCategory,
        price: Number.parseFloat(formData.price) || 0,
        image: formData.image,
        buyLink: formData.buyLink,
        likes: formData.likes,
        dislikes: formData.dislikes,
        rating: calculatedRating,
        kaliteliHammadde: formData.kaliteliHammadde,
        yerliMarka: formData.yerliMarka,
        icerik: formData.ingredients
          .split(",")
          .map((i) => i.trim())
          .filter(Boolean),
        usage: formData.usage,
        evaluation: {
          temiz_icerik: cleanScore,
          formul_kalitesi: Number.parseInt(formData.evaluation.formul_kalitesi) || 0,
          uretim_tesisi: Number.parseInt(formData.evaluation.uretim_tesisi) || 0,
          kullanici_yorumlari: Number.parseInt(formData.evaluation.kullanici_yorumlari) || 0,
          uretim_sertifika: certScore,
          ai_guven_skoru: aiScore,
          dradvice: formData.dradvice,
          certFactors: formData.certFactors,
        },
        cleanFactors: formData.cleanFactors,
        createdAt: product?.createdAt || new Date(),
        influencerCommission: formData.influencerCommission,
      }

      const savedId = await saveProduct(productData)
      console.log("Ürün kaydedildi:", savedId)

      onSave({ ...productData, id: savedId })
    } catch (error) {
      console.error("Ürün kaydedilirken hata:", error)
      alert("Ürün kaydedilirken bir hata oluştu.")
    } finally {
      setSaving(false)
    }
  }

  const availableSubCategories = categories.find(cat => cat.id === formData.category)?.subCategories.map(sc => sc.name) || []

  // Kayıt no normalize fonksiyonu
  const normalizeKayitNo = (kod: string) => (kod || '').replace(/[^a-zA-Z0-9]/g, '').toLowerCase();

  // Ürünün markası
  const productBrand = allBrands.find(b => {
    if (!product) return false;
    if (typeof product.brand === 'string') return b.name.trim().toLowerCase() === product.brand.trim().toLowerCase();
    if (typeof product.brand === 'object' && product.brand !== null) return b.id === product.brand.id || b.name.trim().toLowerCase() === product.brand.name.trim().toLowerCase();
    return false;
  });

  // O markanın diğer ürünleri (bu ürün hariç)
  const brandProducts = products.filter(p => {
    if (!productBrand) return false;
    if (typeof p.brand === 'string') return p.brand.trim().toLowerCase() === productBrand.name.trim().toLowerCase();
    if (typeof p.brand === 'object' && p.brand !== null) return p.brand.id === productBrand.id || p.brand.name.trim().toLowerCase() === productBrand.name.trim().toLowerCase();
    return false;
  }).filter(p => p.id !== product?.id);

  // Ürünün üretici kayıt no'su
  const productKayitNo = normalizeKayitNo((product as any)?.uretimTesisiKayitNo || (product as any)?.["Üretici İşletme Kayıt/Onay Numarası"] || '');
  // Bu kayıt no'ya sahip tesisler
  const relatedFacilities = facilities.filter(fac => productKayitNo && normalizeKayitNo((fac as any).kayit_no || '') === productKayitNo);

  return (
    <Dialog open={true} onOpenChange={onClose}>
      <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>{product ? "Ürün Düzenle" : "Yeni Ürün Ekle"}</DialogTitle>
        </DialogHeader>

        <form onSubmit={handleSubmit} className="space-y-6">
          {/* Basic Information */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Temel Bilgiler</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="name">Ürün Adı *</Label>
                  <Input
                    id="name"
                    value={formData.name}
                    onChange={(e) => {
                      const newValue = e.target.value;
                      if (formData.name !== newValue) {
                        setFormData(prev => ({ ...prev, name: newValue }));
                      }
                    }}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="brand">Marka *</Label>
                  <Popover open={brandSearchOpen} onOpenChange={setBrandSearchOpen}>
                    <PopoverTrigger asChild>
                      <Button
                        variant="outline"
                        role="combobox"
                        aria-expanded={brandSearchOpen}
                        className="w-full justify-between"
                      >
                        {formData.brand && formData.brand.name
                          ? formData.brand.name
                          : "Marka seçin..."}
                        <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                      </Button>
                    </PopoverTrigger>
                    <PopoverContent className="w-full p-0">
                      <Command>
                        <CommandInput placeholder="Marka ara..." />
                        <CommandList>
                          <CommandEmpty>Marka bulunamadı.</CommandEmpty>
                          <CommandGroup>
                            {(allBrands as Brand[]).map((brand: Brand) => (
                              <CommandItem
                                key={brand.id}
                                value={brand.name}
                                onSelect={() => {
                                  if (formData.brand.id !== brand.id) {
                                    setFormData({ ...formData, brand: { id: brand.id, name: brand.name } });
                                  }
                                  setBrandSearchOpen(false);
                                }}
                              >
                                <Check
                                  className={cn(
                                    "mr-2 h-4 w-4",
                                    formData.brand && formData.brand.id === brand.id ? "opacity-100" : "opacity-0"
                                  )}
                                />
                                {brand.name}
                              </CommandItem>
                            ))}
                          </CommandGroup>
                        </CommandList>
                      </Command>
                    </PopoverContent>
                  </Popover>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="category">Kategori *</Label>
                  <Select
                    value={formData.category}
                    onValueChange={(value) => {
                      if (formData.category !== value) {
                        setFormData({ ...formData, category: value, subCategory: "" });
                      }
                    }}
                    disabled={categories.length === 0}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder={categories.length === 0 ? "Kategori bulunamadı" : "Kategori seçin"} />
                    </SelectTrigger>
                    <SelectContent>
                      {categories.length === 0 ? (
                        <div className="px-4 py-2 text-gray-400">Kategori bulunamadı</div>
                      ) : (
                        categories.map((category) => (
                          <SelectItem key={category.id} value={category.id}>
                            {category.name}
                          </SelectItem>
                        ))
                      )}
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="subCategory">Alt Kategori</Label>
                  <Select
                    value={formData.subCategory}
                    onValueChange={(value) => {
                      if (formData.subCategory !== value) {
                        setFormData({ ...formData, subCategory: value });
                      }
                    }}
                    disabled={!formData.category}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder={!formData.category ? "İlk önce bir Ana kategori seçin" : "Alt kategori seçin"} />
                    </SelectTrigger>
                    <SelectContent>
                      {!formData.category ? (
                        <div className="px-4 py-2 text-gray-400">İlk önce bir Ana kategori seçin</div>
                      ) : (
                        availableSubCategories.map((subCategory) => (
                          <SelectItem key={subCategory} value={subCategory}>
                          {subCategory}
                          </SelectItem>
                        ))
                      )}
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="price">Fiyat (TL) *</Label>
                  <Input
                    id="price"
                    type="number"
                    step="0.01"
                    value={formData.price}
                    onChange={(e) => {
                      const newValue = e.target.value;
                      if (formData.price !== newValue) {
                        setFormData(prev => ({ ...prev, price: newValue }));
                      }
                    }}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="image">Resim URL</Label>
                  <Input
                    id="image"
                    type="url"
                    value={formData.image}
                    onChange={(e) => {
                      const newValue = e.target.value;
                      if (formData.image !== newValue) {
                        setFormData(prev => ({ ...prev, image: newValue }));
                      }
                    }}
                    placeholder="https://..."
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="buyLink">Satın Alma Linki</Label>
                <Input
                  id="buyLink"
                  type="url"
                  value={formData.buyLink}
                  onChange={(e) => {
                    const newValue = e.target.value;
                    if (formData.buyLink !== newValue) {
                      setFormData(prev => ({ ...prev, buyLink: newValue }));
                    }
                  }}
                  placeholder="https://..."
                />
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="likes">Beğeni Sayısı</Label>
                  <Input
                    id="likes"
                    type="number"
                    value={formData.likes}
                    onChange={(e) => {
                      const newValue = Number.parseInt(e.target.value) || 0;
                      if (formData.likes !== newValue) {
                        setFormData(prev => ({ ...prev, likes: newValue }));
                      }
                    }}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="dislikes">Beğenmeme Sayısı</Label>
                  <Input
                    id="dislikes"
                    type="number"
                    value={formData.dislikes}
                    onChange={(e) => {
                      const newValue = Number.parseInt(e.target.value) || 0;
                      if (formData.dislikes !== newValue) {
                        setFormData(prev => ({ ...prev, dislikes: newValue }));
                      }
                    }}
                  />
                </div>
              </div>

              <div className="flex items-center space-x-2">
                <Checkbox
                  id="kaliteliHammadde"
                  checked={formData.kaliteliHammadde}
                  onCheckedChange={(checked) => {
                    if (formData.kaliteliHammadde !== (checked as boolean)) {
                      setFormData(prev => ({ ...prev, kaliteliHammadde: checked as boolean }));
                    }
                  }}
                />
                <Label htmlFor="kaliteliHammadde" className="text-sm font-medium">
                  Kaliteli Hammadde (Güven Skoru Yeşil)
                </Label>
              </div>

              <div className="flex items-center space-x-2">
                <Checkbox
                  id="yerliMarka"
                  checked={formData.yerliMarka}
                  onCheckedChange={(checked) => {
                    if (formData.yerliMarka !== (checked as boolean)) {
                      setFormData(prev => ({ ...prev, yerliMarka: checked as boolean }));
                    }
                  }}
                />
                <Label htmlFor="yerliMarka" className="text-sm font-medium">
                  Yerli Marka (Yerli ve Milli Rozeti)
                </Label>
              </div>
            </CardContent>
          </Card>

          {/* Evaluation Scores */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Değerlendirme Skorları</CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              {/* AI Safety Score */}
              <div>
                <div className="flex items-center justify-between mb-2">
                  <span className="font-medium">🤖 AI Güven Skoru</span>
                  <span className="text-2xl font-bold text-purple-600">{aiScore}%</span>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-3">
                  <div
                    className="bg-gradient-to-r from-purple-500 to-blue-500 h-3 rounded-full transition-all duration-500"
                    style={{ width: `${aiScore}%` }}
                  />
                </div>
                <div className="mt-3">
                  <Label htmlFor="ai-score" className="text-sm font-medium">
                    Manuel AI Güven Skoru (0-100)
                  </Label>
                  <Input
                    id="ai-score"
                    type="number"
                    min="0"
                    max="100"
                    value={formData.evaluation.ai_guven_skoru}
                    onChange={(e) => {
                      const newValue = e.target.value;
                      if (formData.evaluation.ai_guven_skoru !== newValue) {
                        setFormData(prev => ({
                          ...prev,
                          evaluation: {
                            ...prev.evaluation,
                            ai_guven_skoru: newValue
                          }
                        }));
                      }
                    }}
                    placeholder="85"
                    className="mt-1"
                  />
                </div>
                <p className="text-sm text-gray-600 mt-2">
                  Bu skor, ürünün sahte yorumlardan ne kadar temiz olduğunu gösterir. Manuel olarak belirleyin.
                </p>
              </div>

              {/* Clean Content Factors */}
              <div>
                <h4 className="font-medium mb-3">Temiz İçerik Faktörleri</h4>
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                  <div className="space-y-2">
                    <Label className="text-sm font-medium">Form Tipi (Sadece birini seçin)</Label>
                    {Object.entries(formData.cleanFactors.form).map(([key, value]) => (
                      <div key={key} className="flex items-center space-x-2">
                        <Checkbox
                          id={`form-${key}`}
                          checked={value}
                          onCheckedChange={(checked) => {
                            // Radio button mantığı - sadece seçilen true, diğerleri false
                            const newForm = Object.keys(formData.cleanFactors.form).reduce((acc, formKey) => {
                              acc[formKey as keyof typeof formData.cleanFactors.form] = formKey === key ? (checked as boolean) : false
                              return acc
                            }, {} as typeof formData.cleanFactors.form)
                            
                            // Sadece değişiklik varsa güncelle
                            if (JSON.stringify(formData.cleanFactors.form) !== JSON.stringify(newForm)) {
                              setFormData(prev => ({
                                ...prev,
                                cleanFactors: {
                                  ...prev.cleanFactors,
                                  form: newForm,
                                },
                              }));
                            }
                          }}
                        />
                        <Label htmlFor={`form-${key}`} className="text-sm capitalize">
                          {key}
                        </Label>
                      </div>
                    ))}
                  </div>

                  <div className="space-y-2">
                    <Label className="text-sm font-medium">Katkı Maddeleri</Label>
                    {Object.entries(formData.cleanFactors.additives).map(([key, value]) => (
                      <div key={key} className="flex items-center space-x-2">
                        <Checkbox
                          id={`additive-${key}`}
                          checked={value}
                          onCheckedChange={(checked) => {
                            const newAdditives = { ...formData.cleanFactors.additives, [key]: checked as boolean };
                            if (JSON.stringify(formData.cleanFactors.additives) !== JSON.stringify(newAdditives)) {
                              setFormData(prev => ({
                                ...prev,
                                cleanFactors: {
                                  ...prev.cleanFactors,
                                  additives: newAdditives,
                                },
                              }));
                            }
                          }}
                        />
                        <Label htmlFor={`additive-${key}`} className="text-sm capitalize">
                          {key}
                        </Label>
                      </div>
                    ))}
                  </div>

                  <div className="space-y-2">
                    <Label className="text-sm font-medium">Temiz İçerik Skoru</Label>
                    <div className="text-2xl font-bold text-green-600">{cleanScore}%</div>
                    <div className="w-full bg-gray-200 rounded-full h-2">
                      <div
                        className="bg-green-500 h-2 rounded-full transition-all duration-300"
                        style={{ width: `${cleanScore}%` }}
                      />
                    </div>
                  </div>
                </div>
              </div>

              {/* Certificates */}
              <div>
                <h4 className="font-medium mb-3">Sertifikalar</h4>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                  {Object.entries(formData.certFactors).map(([key, value]) => (
                    <div key={key} className="flex items-center space-x-2">
                      <Checkbox
                        id={`cert-${key}`}
                        checked={value}
                        onCheckedChange={(checked) => {
                          const newCertFactors = { ...formData.certFactors, [key]: checked as boolean };
                          if (JSON.stringify(formData.certFactors) !== JSON.stringify(newCertFactors)) {
                            setFormData(prev => ({
                              ...prev,
                              certFactors: newCertFactors,
                            }));
                          }
                        }}
                      />
                      <Label htmlFor={`cert-${key}`} className="text-sm uppercase">
                        {key}
                      </Label>
                    </div>
                  ))}
                </div>
                <div className="mt-3">
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium">Sertifika Skoru</span>
                    <span className="text-lg font-bold text-blue-600">{certScore}%</span>
                  </div>
                  <div className="w-full bg-gray-200 rounded-full h-2 mt-1">
                    <div
                      className="bg-blue-500 h-2 rounded-full transition-all duration-300"
                      style={{ width: `${certScore}%` }}
                    />
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Additional Information */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Ek Bilgiler</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="ingredients">İçerikler (virgülle ayırın)</Label>
                <Input
                  id="ingredients"
                  value={formData.ingredients}
                  onChange={(e) => {
                    const newValue = e.target.value;
                    if (formData.ingredients !== newValue) {
                      setFormData(prev => ({ ...prev, ingredients: newValue }));
                    }
                  }}
                  placeholder="B6 Vitamini, D3 Vitamini, Çinko"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="usage">Kullanım Talimatı</Label>
                <Textarea
                  id="usage"
                  value={formData.usage}
                  onChange={(e) => {
                    const newValue = e.target.value;
                    if (formData.usage !== newValue) {
                      setFormData(prev => ({ ...prev, usage: newValue }));
                    }
                  }}
                  placeholder="Günde 1 kapsül, yemekle birlikte alınız."
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="buyLink">Satın Alma Linki</Label>
                <Input
                  id="buyLink"
                  type="url"
                  value={formData.buyLink}
                  onChange={(e) => {
                    const newValue = e.target.value;
                    if (formData.buyLink !== newValue) {
                      setFormData(prev => ({ ...prev, buyLink: newValue }));
                    }
                  }}
                  placeholder="https://..."
                />
              </div>
            </CardContent>
          </Card>

          {/* Ürün bazında Influencer Komisyon Oranı */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Influencer Komisyon Oranı (Ürün Bazında)</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2">
              {formData.influencerCommission.map((ic, idx) => (
                <div key={idx} className="flex items-center gap-2 mb-2">
                  <Select
                    value={ic.platform}
                    onValueChange={val => {
                      setFormData(prevFormData => {
                        const updatedInfluencerCommission = prevFormData.influencerCommission.map((item, itemIdx) => 
                          itemIdx === idx ? { ...item, platform: val } : item
                        );
                        if (JSON.stringify(prevFormData.influencerCommission) !== JSON.stringify(updatedInfluencerCommission)) {
                          return { ...prevFormData, influencerCommission: updatedInfluencerCommission };
                        }
                        return prevFormData;
                      });
                    }}
                  >
                    <SelectTrigger className="w-32">
                      <SelectValue placeholder="Platform" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="Trendyol">Trendyol</SelectItem>
                      <SelectItem value="Hepsiburada">Hepsiburada</SelectItem>
                      <SelectItem value="Yerel Site">Yerel Site</SelectItem>
                    </SelectContent>
                  </Select>
                  <Input
                    type="number"
                    min={0}
                    max={100}
                    step={0.01}
                    className="w-24"
                    value={ic.rate}
                    onChange={e => {
                      const newRate = parseFloat(e.target.value);
                      setFormData(prevFormData => {
                        const updated = [...prevFormData.influencerCommission];
                        if (updated[idx].rate !== newRate) {
                          updated[idx].rate = newRate;
                          return { ...prevFormData, influencerCommission: updated };
                        }
                        return prevFormData;
                      });
                    }}
                    placeholder="%"
                  />
                  <Button type="button" variant="destructive" size="icon" onClick={() => {
                    setFormData(prevFormData => {
                      const updatedCommission = prevFormData.influencerCommission.filter((_, i) => i !== idx);
                      if (JSON.stringify(prevFormData.influencerCommission) !== JSON.stringify(updatedCommission)) {
                        return { ...prevFormData, influencerCommission: updatedCommission };
                      }
                      return prevFormData;
                    });
                  }}>Sil</Button>
                </div>
              ))}
              <Button type="button" variant="outline" size="sm" onClick={() => {
                setFormData({ ...formData, influencerCommission: [...formData.influencerCommission, { platform: "Trendyol", rate: 0 }] })
              }}>+ Platform Ekle</Button>
            </CardContent>
          </Card>

          {/* Form Actions */}
          <div className="flex justify-end gap-4">
            <Button type="button" variant="outline" onClick={onClose}>
              İptal
            </Button>
            <Button type="submit" disabled={saving}>
              <Sparkles className="h-4 w-4 mr-2" />
              {saving ? "Kaydediliyor..." : product ? "Güncelle" : "Kaydet"}
            </Button>
          </div>
        </form>
        {/* Ürünün ait olduğu marka ve diğer ürünleri */}
        {productBrand && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold mb-2">Marka: {productBrand.name}</h3>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
              {brandProducts.length === 0 && <div className="text-gray-400">Bu markanın başka ürünü yok.</div>}
              {brandProducts.map((p) => (
                <div key={p.id} className="flex items-center gap-2 bg-gray-50 border rounded p-2 min-h-[64px]">
                  <div className="w-14 h-14 flex-shrink-0 bg-white rounded flex items-center justify-center overflow-hidden">
                    <Image src={p.image || "/placeholder.svg"} alt={p.name} width={56} height={56} className="object-contain" />
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="font-medium text-sm truncate" title={p.name}>{p.name}</div>
                    <div className="flex flex-col gap-0.5 mt-1">
                      <span className="text-xs text-blue-700">AI Güven: <b>{getAISafetyScore(p)}%</b></span>
                      <span className="text-xs text-green-700">Temiz İçerik: <b>{getCleanContentScore(p)}%</b></span>
                      <span className="text-xs text-yellow-700">Sertifika: <b>{getCertificateScore(p)}%</b></span>
                      <span className="text-xs text-purple-700">Popülerlik: <b>{p.likes + p.dislikes > 0 ? Math.round((p.likes / (p.likes + p.dislikes)) * 100) : 'N/A'}%</b></span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}
        {/* İlgili üretim tesisleri */}
        {relatedFacilities.length > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold mb-2">İlgili Üretim Tesisleri</h3>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
              {relatedFacilities.map(fac => (
                <div key={fac.id} className="flex items-center gap-2 bg-gray-50 border rounded p-2 min-h-[64px]">
                  <div className="w-12 h-12 flex-shrink-0 bg-white rounded flex items-center justify-center overflow-hidden">
                    <Image src={fac.logoUrl || "/placeholder-logo.png"} alt={fac.name} width={48} height={48} className="object-contain" />
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="font-medium text-sm truncate" title={fac.name}>{fac.name}</div>
                    <div className="text-xs text-gray-600">Kayıt No: {(fac as any).kayit_no || '-'}</div>
                    <div className="text-xs text-yellow-700">Puan: <b>{fac.score}</b></div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}
      </DialogContent>
    </Dialog>
  )
} 